<?php

use MediaWiki\Auth\AbstractPreAuthenticationProvider;
use MediaWiki\Auth\AuthenticationRequest;
use MediaWiki\Context\DerivativeContext;
use MediaWiki\Context\IContextSource;
use MediaWiki\Context\RequestContext;
use MediaWiki\Exception\ErrorPageError;
use MediaWiki\MainConfigNames;
use MediaWiki\Request\FauxRequest;
use MediaWiki\SpecialPage\SpecialPage;
use MediaWiki\Specials\SpecialCreateAccount;
use Wikimedia\Parsoid\Utils\DOMUtils;

/**
 * @covers \MediaWiki\Specials\SpecialCreateAccount
 * @group Database
 */
class SpecialCreateAccountTest extends SpecialPageTestBase {
	/**
	 * @inheritDoc
	 */
	protected function newSpecialPage( ?IContextSource $context = null ) {
		$services = $this->getServiceContainer();
		$context ??= RequestContext::getMain();
		$page = new SpecialCreateAccount(
			$services->getAuthManager(),
			$services->getFormatterFactory(),
			$services->getUserIdentityUtils()
		);
		$page->setContext( $context );
		$context->setTitle( $page->getPageTitle() );
		return $page;
	}

	public function testCheckPermissions() {
		$readOnlyMode = $this->getServiceContainer()->getReadOnlyMode();
		$readOnlyMode->setReason( 'Test' );

		$this->expectException( ErrorPageError::class );
		$specialPage = $this->newSpecialPage();
		$specialPage->checkPermissions();
	}

	/**
	 * Regression test for T360717 -- missing hidden fields from Special:CreateAccount
	 */
	public function testHiddenField() {
		$config = $this->getServiceContainer()->getMainConfig()->get( MainConfigNames::AuthManagerConfig );
		$config['preauth']['MockAuthProviderWithHiddenField'] = [
			'class' => MockAuthProviderWithHiddenField::class
		];
		$this->overrideConfigValue( MainConfigNames::AuthManagerConfig, $config );
		$specialPage = $this->newSpecialPage();
		$specialPage->execute( null );
		$html = $specialPage->getOutput()->getHTML();
		$this->assertStringContainsString(
			'<input id="mw-input-captchaId" name="captchaId" type="hidden" value="T360717">',
			$html
		);
	}

	public function testShouldShowTemporaryPasswordAndCreationReasonFieldsForRegisteredUser(): void {
		$user = $this->getTestUser()->getUser();

		$context = new DerivativeContext( RequestContext::getMain() );
		$context->setUser( $user );

		$specialPage = $this->newSpecialPage( $context );
		$specialPage->execute( null );
		$doc = self::getOutputHtml( $specialPage );

		$this->assertNotNull( $doc->getElementById( 'wpReason' ) );
		$this->assertNotNull( $doc->getElementById( 'wpCreateaccountMail' ) );
	}

	public function testShouldNotShowTemporaryPasswordAndCreationReasonFieldsForTempUser(): void {
		$req = new FauxRequest();
		$tempUser = $this->getServiceContainer()
			->getTempUserCreator()
			->create( null, $req )
			->getUser();

		$context = new DerivativeContext( RequestContext::getMain() );
		$context->setUser( $tempUser );

		$specialPage = $this->newSpecialPage( $context );
		$specialPage->execute( null );
		$doc = self::getOutputHtml( $specialPage );

		$this->assertNull(
			$doc->getElementById( 'wpReason' ),
			'Temporary users should not have to provide a reason for their account creation (T328718)'
		);
		$this->assertNull(
			$doc->getElementById( 'wpCreateaccountMail' ),
			'Temporary users should not have the option to have a temporary password sent on signup (T328718)'
		);
	}

	/**
	 * Convenience function to get the parsed DOM of the HTML generated by the given special page.
	 * @param SpecialPage $page
	 * @return DOMDocument
	 */
	private static function getOutputHtml( SpecialPage $page ): DOMDocument {
		return DOMUtils::parseHTML( $page->getOutput()->getHTML() );
	}
}

class MockAuthRequestWithHiddenField extends AuthenticationRequest {
	public function getFieldInfo() {
		return [
			'captchaId' => [
				'type' => 'hidden',
				'value' => 'T360717',
				'label' => '',
				'help' => '',
			],
		];
	}
}

class MockAuthProviderWithHiddenField extends AbstractPreAuthenticationProvider {
	public function getAuthenticationRequests( $action, array $options ) {
		return [ new MockAuthRequestWithHiddenField ];
	}
}
