/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package org.apache.vysper.xmpp.addressing.stringprep;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.vysper.compliance.SpecCompliant;

/**
 * This auxiliary class contains all character maps/sets specified by RFC3454.
 * 
 * see http://www.ietf.org/rfc/rfc3454.txt
 * @author Gerolf Seitz (gseitz@apache.org)
 * 
 */
public class StringPrepConstants {
    @SpecCompliant(spec = "RFC3454", section = "B.1")
    public static final Map<String, String> B_1_CommonlyMappedtoNothing;

    @SpecCompliant(spec = "RFC3454", section = "B.2")
    public static final Map<String, String> B_2_MappingForCaseFoldingUsedWithKFC;

    public static final Map<String, String> B_3_MappingForCaseFoldingWithNoNormalization;

    @SpecCompliant(spec = "RFC3454", section = "C.1.1")
    public static final Set<String> C_1_1_AsciiSpaceCharacters;

    @SpecCompliant(spec = "RFC3454", section = "C.1.2")
    public static final Set<String> C_1_2_NonAsciiSpaceCharacters;

    @SpecCompliant(spec = "RFC3454", section = "C.2.1")
    public static final Set<String> C_2_1_AsciiControlCharacters;

    @SpecCompliant(spec = "RFC3454", section = "C.2.2")
    public static final Set<String> C_2_2_NonAsciiControlCharacters;

    @SpecCompliant(spec = "RFC3454", section = "C.3")
    public static final Set<String> C_3_PrivateUse;

    @SpecCompliant(spec = "RFC3454", section = "C.4")
    public static final Set<String> C_4_NonCharacterCodePoints;

    @SpecCompliant(spec = "RFC3454", section = "C.5")
    public static final Set<String> C_5_SurrogateCodes;

    @SpecCompliant(spec = "RFC3454", section = "C.6")
    public static final Set<String> C_6_InappropriateForPlainText;

    @SpecCompliant(spec = "RFC3454", section = "C.7")
    public static final Set<String> C_7_InappropriateForCanonicalRepresentation;

    @SpecCompliant(spec = "RFC3454", section = "C.8")
    public static final Set<String> C_8_ChangeDisplayPropertiesOrAreDeprecated;

    @SpecCompliant(spec = "RFC3454", section = "C.9")
    public static final Set<String> C_9_TaggingCharacters;

    @SpecCompliant(spec = "RFC3454", section = "D.1")
    public static final Set<String> D_1_CharactersWithBiDiPropertiesRorAl;

    @SpecCompliant(spec = "RFC3454", section = "D.2")
    public static final Set<String> D_2_CharactersWithBiDiPropertyL;

    static {
        B_1_CommonlyMappedtoNothing = build_B_1_CommonlyMappedtoNothing();
        B_2_MappingForCaseFoldingUsedWithKFC = build_B_2_MappingForCaseFoldingUsedWithKFC();
        B_3_MappingForCaseFoldingWithNoNormalization = build_B_3_MappingForCaseFoldingWithNoNormalization();
        C_1_1_AsciiSpaceCharacters = build_C_1_1_AsciiSpaceCharacters();
        C_1_2_NonAsciiSpaceCharacters = build_C_1_2_NonAsciiSpaceCharacters();
        C_2_1_AsciiControlCharacters = build_C_2_1_AsciiControlCharacters();
        C_2_2_NonAsciiControlCharacters = build_C_2_2_NonAsciiControlCharacters();
        C_3_PrivateUse = build_C_3_PrivateUse();
        C_4_NonCharacterCodePoints = build_C_4_NonCharacterCodePoints();
        C_5_SurrogateCodes = build_C_5_SurrogateCodes();
        C_6_InappropriateForPlainText = build_C_6_InappropriateForPlainText();
        C_7_InappropriateForCanonicalRepresentation = build_C_7_InappropriateForCanonicalRepresentation();
        C_8_ChangeDisplayPropertiesOrAreDeprecated = build_C_8_ChangeDisplayPropertiesOrAreDeprecated();
        C_9_TaggingCharacters = build_C_9_TaggingCharacters();
        D_1_CharactersWithBiDiPropertiesRorAl = build_D_1_CharactersWithBiDiPropertiesRorAl();
        D_2_CharactersWithBiDiPropertyL = build_D_2_CharactersWithBiDiPropertyL();
    }

    private static final Map<String, String> build_B_1_CommonlyMappedtoNothing() {
        Map<String, String> b_1 = new HashMap<String, String>();
        b_1.put("\u00AD", "");
        b_1.put("\u034F", "");
        b_1.put("\u1806", "");
        b_1.put("\u180B", "");
        b_1.put("\u180C", "");
        b_1.put("\u180D", "");
        b_1.put("\u200B", "");
        b_1.put("\u200C", "");
        b_1.put("\u200D", "");
        b_1.put("\u2060", "");
        b_1.put("\uFE00", "");
        b_1.put("\uFE01", "");
        b_1.put("\uFE02", "");
        b_1.put("\uFE03", "");
        b_1.put("\uFE04", "");
        b_1.put("\uFE05", "");
        b_1.put("\uFE06", "");
        b_1.put("\uFE07", "");
        b_1.put("\uFE08", "");
        b_1.put("\uFE09", "");
        b_1.put("\uFE0A", "");
        b_1.put("\uFE0B", "");
        b_1.put("\uFE0C", "");
        b_1.put("\uFE0D", "");
        b_1.put("\uFE0E", "");
        b_1.put("\uFE0F", "");
        b_1.put("\uFEFF", "");
        b_1.put("\u2060", "");
        b_1.put("\uFE00", "");
        b_1.put("\uFE01", "");
        b_1.put("\uFE02", "");
        b_1.put("\uFE03", "");
        b_1.put("\uFE04", "");
        b_1.put("\uFE05", "");
        b_1.put("\uFE06", "");
        b_1.put("\uFE07", "");
        b_1.put("\uFE08", "");
        b_1.put("\uFE09", "");
        b_1.put("\uFE0A", "");
        b_1.put("\uFE0B", "");
        b_1.put("\uFE0C", "");
        b_1.put("\uFE0D", "");
        b_1.put("\uFE0E", "");
        b_1.put("\uFE0F", "");
        b_1.put("\uFEFF", "");
        return Collections.unmodifiableMap(b_1);
    }

    private static final Map<String, String> build_B_2_MappingForCaseFoldingUsedWithKFC() {
        Map<String, String> b_2 = new HashMap<String, String>();
        b_2.put("\u0041", "\u0061");
        b_2.put("\u0042", "\u0062");
        b_2.put("\u0043", "\u0063");
        b_2.put("\u0044", "\u0064");
        b_2.put("\u0045", "\u0065");
        b_2.put("\u0046", "\u0066");
        b_2.put("\u0047", "\u0067");
        b_2.put("\u0048", "\u0068");
        b_2.put("\u0049", "\u0069");
        b_2.put("\u004A", "\u006A");
        b_2.put("\u004B", "\u006B");
        b_2.put("\u004C", "\u006C");
        b_2.put("\u004D", "\u006D");
        b_2.put("\u004E", "\u006E");
        b_2.put("\u004F", "\u006F");
        b_2.put("\u0050", "\u0070");
        b_2.put("\u0051", "\u0071");
        b_2.put("\u0052", "\u0072");
        b_2.put("\u0053", "\u0073");
        b_2.put("\u0054", "\u0074");
        b_2.put("\u0055", "\u0075");
        b_2.put("\u0056", "\u0076");
        b_2.put("\u0057", "\u0077");
        b_2.put("\u0058", "\u0078");
        b_2.put("\u0059", "\u0079");
        b_2.put("\u005A", "\u007A");
        b_2.put("\u00B5", "\u03BC");
        b_2.put("\u00C0", "\u00E0");
        b_2.put("\u00C1", "\u00E1");
        b_2.put("\u00C2", "\u00E2");
        b_2.put("\u00C3", "\u00E3");
        b_2.put("\u00C4", "\u00E4");
        b_2.put("\u00C5", "\u00E5");
        b_2.put("\u00C6", "\u00E6");
        b_2.put("\u00C7", "\u00E7");
        b_2.put("\u00C8", "\u00E8");
        b_2.put("\u00C9", "\u00E9");
        b_2.put("\u00CA", "\u00EA");
        b_2.put("\u00CB", "\u00EB");
        b_2.put("\u00CC", "\u00EC");
        b_2.put("\u00CD", "\u00ED");
        b_2.put("\u00CE", "\u00EE");
        b_2.put("\u00CF", "\u00EF");
        b_2.put("\u00D0", "\u00F0");
        b_2.put("\u00D1", "\u00F1");
        b_2.put("\u00D2", "\u00F2");
        b_2.put("\u00D3", "\u00F3");
        b_2.put("\u00D4", "\u00F4");
        b_2.put("\u00D5", "\u00F5");
        b_2.put("\u00D6", "\u00F6");
        b_2.put("\u00D8", "\u00F8");
        b_2.put("\u00D9", "\u00F9");
        b_2.put("\u00DA", "\u00FA");
        b_2.put("\u00DB", "\u00FB");
        b_2.put("\u00DC", "\u00FC");
        b_2.put("\u00DD", "\u00FD");
        b_2.put("\u00DE", "\u00FE");
        b_2.put("\u00DF", "\u0073\u0073");
        b_2.put("\u0100", "\u0101");
        b_2.put("\u0102", "\u0103");
        b_2.put("\u0104", "\u0105");
        b_2.put("\u0106", "\u0107");
        b_2.put("\u0108", "\u0109");
        b_2.put("\u010A", "\u010B");
        b_2.put("\u010C", "\u010D");
        b_2.put("\u010E", "\u010F");
        b_2.put("\u0110", "\u0111");
        b_2.put("\u0112", "\u0113");
        b_2.put("\u0114", "\u0115");
        b_2.put("\u0116", "\u0117");
        b_2.put("\u0118", "\u0119");
        b_2.put("\u011A", "\u011B");
        b_2.put("\u011C", "\u011D");
        b_2.put("\u011E", "\u011F");
        b_2.put("\u0120", "\u0121");
        b_2.put("\u0122", "\u0123");
        b_2.put("\u0124", "\u0125");
        b_2.put("\u0126", "\u0127");
        b_2.put("\u0128", "\u0129");
        b_2.put("\u012A", "\u012B");
        b_2.put("\u012C", "\u012D");
        b_2.put("\u012E", "\u012F");
        b_2.put("\u0130", "\u0069\u0307");
        b_2.put("\u0132", "\u0133");
        b_2.put("\u0134", "\u0135");
        b_2.put("\u0136", "\u0137");
        b_2.put("\u0139", "\u013A");
        b_2.put("\u013B", "\u013C");
        b_2.put("\u013D", "\u013E");
        b_2.put("\u013F", "\u0140");
        b_2.put("\u0141", "\u0142");
        b_2.put("\u0143", "\u0144");
        b_2.put("\u0145", "\u0146");
        b_2.put("\u0147", "\u0148");
        b_2.put("\u0149", "\u02BC\u006E");
        b_2.put("\u014A", "\u014B");
        b_2.put("\u014C", "\u014D");
        b_2.put("\u014E", "\u014F");
        b_2.put("\u0150", "\u0151");
        b_2.put("\u0152", "\u0153");
        b_2.put("\u0154", "\u0155");
        b_2.put("\u0156", "\u0157");
        b_2.put("\u0158", "\u0159");
        b_2.put("\u015A", "\u015B");
        b_2.put("\u015C", "\u015D");
        b_2.put("\u015E", "\u015F");
        b_2.put("\u0160", "\u0161");
        b_2.put("\u0162", "\u0163");
        b_2.put("\u0164", "\u0165");
        b_2.put("\u0166", "\u0167");
        b_2.put("\u0168", "\u0169");
        b_2.put("\u016A", "\u016B");
        b_2.put("\u016C", "\u016D");
        b_2.put("\u016E", "\u016F");
        b_2.put("\u0170", "\u0171");
        b_2.put("\u0172", "\u0173");
        b_2.put("\u0174", "\u0175");
        b_2.put("\u0176", "\u0177");
        b_2.put("\u0178", "\u00FF");
        b_2.put("\u0179", "\u017A");
        b_2.put("\u017B", "\u017C");
        b_2.put("\u017D", "\u017E");
        b_2.put("\u017F", "\u0073");
        b_2.put("\u0181", "\u0253");
        b_2.put("\u0182", "\u0183");
        b_2.put("\u0184", "\u0185");
        b_2.put("\u0186", "\u0254");
        b_2.put("\u0187", "\u0188");
        b_2.put("\u0189", "\u0256");
        b_2.put("\u018A", "\u0257");
        b_2.put("\u018B", "\u018C");
        b_2.put("\u018E", "\u01DD");
        b_2.put("\u018F", "\u0259");
        b_2.put("\u0190", "\u025B");
        b_2.put("\u0191", "\u0192");
        b_2.put("\u0193", "\u0260");
        b_2.put("\u0194", "\u0263");
        b_2.put("\u0196", "\u0269");
        b_2.put("\u0197", "\u0268");
        b_2.put("\u0198", "\u0199");
        b_2.put("\u019C", "\u026F");
        b_2.put("\u019D", "\u0272");
        b_2.put("\u019F", "\u0275");
        b_2.put("\u01A0", "\u01A1");
        b_2.put("\u01A2", "\u01A3");
        b_2.put("\u01A4", "\u01A5");
        b_2.put("\u01A6", "\u0280");
        b_2.put("\u01A7", "\u01A8");
        b_2.put("\u01A9", "\u0283");
        b_2.put("\u01AC", "\u01AD");
        b_2.put("\u01AE", "\u0288");
        b_2.put("\u01AF", "\u01B0");
        b_2.put("\u01B1", "\u028A");
        b_2.put("\u01B2", "\u028B");
        b_2.put("\u01B3", "\u01B4");
        b_2.put("\u01B5", "\u01B6");
        b_2.put("\u01B7", "\u0292");
        b_2.put("\u01B8", "\u01B9");
        b_2.put("\u01BC", "\u01BD");
        b_2.put("\u01C4", "\u01C6");
        b_2.put("\u01C5", "\u01C6");
        b_2.put("\u01C7", "\u01C9");
        b_2.put("\u01C8", "\u01C9");
        b_2.put("\u01CA", "\u01CC");
        b_2.put("\u01CB", "\u01CC");
        b_2.put("\u01CD", "\u01CE");
        b_2.put("\u01CF", "\u01D0");
        b_2.put("\u01D1", "\u01D2");
        b_2.put("\u01D3", "\u01D4");
        b_2.put("\u01D5", "\u01D6");
        b_2.put("\u01D7", "\u01D8");
        b_2.put("\u01D9", "\u01DA");
        b_2.put("\u01DB", "\u01DC");
        b_2.put("\u01DE", "\u01DF");
        b_2.put("\u01E0", "\u01E1");
        b_2.put("\u01E2", "\u01E3");
        b_2.put("\u01E4", "\u01E5");
        b_2.put("\u01E6", "\u01E7");
        b_2.put("\u01E8", "\u01E9");
        b_2.put("\u01EA", "\u01EB");
        b_2.put("\u01EC", "\u01ED");
        b_2.put("\u01EE", "\u01EF");
        b_2.put("\u01F0", "\u006A\u030C");
        b_2.put("\u01F1", "\u01F3");
        b_2.put("\u01F2", "\u01F3");
        b_2.put("\u01F4", "\u01F5");
        b_2.put("\u01F6", "\u0195");
        b_2.put("\u01F7", "\u01BF");
        b_2.put("\u01F8", "\u01F9");
        b_2.put("\u01FA", "\u01FB");
        b_2.put("\u01FC", "\u01FD");
        b_2.put("\u01FE", "\u01FF");
        b_2.put("\u0200", "\u0201");
        b_2.put("\u0202", "\u0203");
        b_2.put("\u0204", "\u0205");
        b_2.put("\u0206", "\u0207");
        b_2.put("\u0208", "\u0209");
        b_2.put("\u020A", "\u020B");
        b_2.put("\u020C", "\u020D");
        b_2.put("\u020E", "\u020F");
        b_2.put("\u0210", "\u0211");
        b_2.put("\u0212", "\u0213");
        b_2.put("\u0214", "\u0215");
        b_2.put("\u0216", "\u0217");
        b_2.put("\u0218", "\u0219");
        b_2.put("\u021A", "\u021B");
        b_2.put("\u021C", "\u021D");
        b_2.put("\u021E", "\u021F");
        b_2.put("\u0220", "\u019E");
        b_2.put("\u0222", "\u0223");
        b_2.put("\u0224", "\u0225");
        b_2.put("\u0226", "\u0227");
        b_2.put("\u0228", "\u0229");
        b_2.put("\u022A", "\u022B");
        b_2.put("\u022C", "\u022D");
        b_2.put("\u022E", "\u022F");
        b_2.put("\u0230", "\u0231");
        b_2.put("\u0232", "\u0233");
        b_2.put("\u0345", "\u03B9");
        b_2.put("\u037A", "\u0020\u03B9");
        b_2.put("\u0386", "\u03AC");
        b_2.put("\u0388", "\u03AD");
        b_2.put("\u0389", "\u03AE");
        b_2.put("\u038A", "\u03AF");
        b_2.put("\u038C", "\u03CC");
        b_2.put("\u038E", "\u03CD");
        b_2.put("\u038F", "\u03CE");
        b_2.put("\u0390", "\u03B9\u0308\u0301");
        b_2.put("\u0391", "\u03B1");
        b_2.put("\u0392", "\u03B2");
        b_2.put("\u0393", "\u03B3");
        b_2.put("\u0394", "\u03B4");
        b_2.put("\u0395", "\u03B5");
        b_2.put("\u0396", "\u03B6");
        b_2.put("\u0397", "\u03B7");
        b_2.put("\u0398", "\u03B8");
        b_2.put("\u0399", "\u03B9");
        b_2.put("\u039A", "\u03BA");
        b_2.put("\u039B", "\u03BB");
        b_2.put("\u039C", "\u03BC");
        b_2.put("\u039D", "\u03BD");
        b_2.put("\u039E", "\u03BE");
        b_2.put("\u039F", "\u03BF");
        b_2.put("\u03A0", "\u03C0");
        b_2.put("\u03A1", "\u03C1");
        b_2.put("\u03A3", "\u03C3");
        b_2.put("\u03A4", "\u03C4");
        b_2.put("\u03A5", "\u03C5");
        b_2.put("\u03A6", "\u03C6");
        b_2.put("\u03A7", "\u03C7");
        b_2.put("\u03A8", "\u03C8");
        b_2.put("\u03A9", "\u03C9");
        b_2.put("\u03AA", "\u03CA");
        b_2.put("\u03AB", "\u03CB");
        b_2.put("\u03B0", "\u03C5\u0308\u0301");
        b_2.put("\u03C2", "\u03C3");
        b_2.put("\u03D0", "\u03B2");
        b_2.put("\u03D1", "\u03B8");
        b_2.put("\u03D2", "\u03C5");
        b_2.put("\u03D3", "\u03CD");
        b_2.put("\u03D4", "\u03CB");
        b_2.put("\u03D5", "\u03C6");
        b_2.put("\u03D6", "\u03C0");
        b_2.put("\u03D8", "\u03D9");
        b_2.put("\u03DA", "\u03DB");
        b_2.put("\u03DC", "\u03DD");
        b_2.put("\u03DE", "\u03DF");
        b_2.put("\u03E0", "\u03E1");
        b_2.put("\u03E2", "\u03E3");
        b_2.put("\u03E4", "\u03E5");
        b_2.put("\u03E6", "\u03E7");
        b_2.put("\u03E8", "\u03E9");
        b_2.put("\u03EA", "\u03EB");
        b_2.put("\u03EC", "\u03ED");
        b_2.put("\u03EE", "\u03EF");
        b_2.put("\u03F0", "\u03BA");
        b_2.put("\u03F1", "\u03C1");
        b_2.put("\u03F2", "\u03C3");
        b_2.put("\u03F4", "\u03B8");
        b_2.put("\u03F5", "\u03B5");
        b_2.put("\u0400", "\u0450");
        b_2.put("\u0401", "\u0451");
        b_2.put("\u0402", "\u0452");
        b_2.put("\u0403", "\u0453");
        b_2.put("\u0404", "\u0454");
        b_2.put("\u0405", "\u0455");
        b_2.put("\u0406", "\u0456");
        b_2.put("\u0407", "\u0457");
        b_2.put("\u0408", "\u0458");
        b_2.put("\u0409", "\u0459");
        b_2.put("\u040A", "\u045A");
        b_2.put("\u040B", "\u045B");
        b_2.put("\u040C", "\u045C");
        b_2.put("\u040D", "\u045D");
        b_2.put("\u040E", "\u045E");
        b_2.put("\u040F", "\u045F");
        b_2.put("\u0410", "\u0430");
        b_2.put("\u0411", "\u0431");
        b_2.put("\u0412", "\u0432");
        b_2.put("\u0413", "\u0433");
        b_2.put("\u0414", "\u0434");
        b_2.put("\u0415", "\u0435");
        b_2.put("\u0416", "\u0436");
        b_2.put("\u0417", "\u0437");
        b_2.put("\u0418", "\u0438");
        b_2.put("\u0419", "\u0439");
        b_2.put("\u041A", "\u043A");
        b_2.put("\u041B", "\u043B");
        b_2.put("\u041C", "\u043C");
        b_2.put("\u041D", "\u043D");
        b_2.put("\u041E", "\u043E");
        b_2.put("\u041F", "\u043F");
        b_2.put("\u0420", "\u0440");
        b_2.put("\u0421", "\u0441");
        b_2.put("\u0422", "\u0442");
        b_2.put("\u0423", "\u0443");
        b_2.put("\u0424", "\u0444");
        b_2.put("\u0425", "\u0445");
        b_2.put("\u0426", "\u0446");
        b_2.put("\u0427", "\u0447");
        b_2.put("\u0428", "\u0448");
        b_2.put("\u0429", "\u0449");
        b_2.put("\u042A", "\u044A");
        b_2.put("\u042B", "\u044B");
        b_2.put("\u042C", "\u044C");
        b_2.put("\u042D", "\u044D");
        b_2.put("\u042E", "\u044E");
        b_2.put("\u042F", "\u044F");
        b_2.put("\u0460", "\u0461");
        b_2.put("\u0462", "\u0463");
        b_2.put("\u0464", "\u0465");
        b_2.put("\u0466", "\u0467");
        b_2.put("\u0468", "\u0469");
        b_2.put("\u046A", "\u046B");
        b_2.put("\u046C", "\u046D");
        b_2.put("\u046E", "\u046F");
        b_2.put("\u0470", "\u0471");
        b_2.put("\u0472", "\u0473");
        b_2.put("\u0474", "\u0475");
        b_2.put("\u0476", "\u0477");
        b_2.put("\u0478", "\u0479");
        b_2.put("\u047A", "\u047B");
        b_2.put("\u047C", "\u047D");
        b_2.put("\u047E", "\u047F");
        b_2.put("\u0480", "\u0481");
        b_2.put("\u048A", "\u048B");
        b_2.put("\u048C", "\u048D");
        b_2.put("\u048E", "\u048F");
        b_2.put("\u0490", "\u0491");
        b_2.put("\u0492", "\u0493");
        b_2.put("\u0494", "\u0495");
        b_2.put("\u0496", "\u0497");
        b_2.put("\u0498", "\u0499");
        b_2.put("\u049A", "\u049B");
        b_2.put("\u049C", "\u049D");
        b_2.put("\u049E", "\u049F");
        b_2.put("\u04A0", "\u04A1");
        b_2.put("\u04A2", "\u04A3");
        b_2.put("\u04A4", "\u04A5");
        b_2.put("\u04A6", "\u04A7");
        b_2.put("\u04A8", "\u04A9");
        b_2.put("\u04AA", "\u04AB");
        b_2.put("\u04AC", "\u04AD");
        b_2.put("\u04AE", "\u04AF");
        b_2.put("\u04B0", "\u04B1");
        b_2.put("\u04B2", "\u04B3");
        b_2.put("\u04B4", "\u04B5");
        b_2.put("\u04B6", "\u04B7");
        b_2.put("\u04B8", "\u04B9");
        b_2.put("\u04BA", "\u04BB");
        b_2.put("\u04BC", "\u04BD");
        b_2.put("\u04BE", "\u04BF");
        b_2.put("\u04C1", "\u04C2");
        b_2.put("\u04C3", "\u04C4");
        b_2.put("\u04C5", "\u04C6");
        b_2.put("\u04C7", "\u04C8");
        b_2.put("\u04C9", "\u04CA");
        b_2.put("\u04CB", "\u04CC");
        b_2.put("\u04CD", "\u04CE");
        b_2.put("\u04D0", "\u04D1");
        b_2.put("\u04D2", "\u04D3");
        b_2.put("\u04D4", "\u04D5");
        b_2.put("\u04D6", "\u04D7");
        b_2.put("\u04D8", "\u04D9");
        b_2.put("\u04DA", "\u04DB");
        b_2.put("\u04DC", "\u04DD");
        b_2.put("\u04DE", "\u04DF");
        b_2.put("\u04E0", "\u04E1");
        b_2.put("\u04E2", "\u04E3");
        b_2.put("\u04E4", "\u04E5");
        b_2.put("\u04E6", "\u04E7");
        b_2.put("\u04E8", "\u04E9");
        b_2.put("\u04EA", "\u04EB");
        b_2.put("\u04EC", "\u04ED");
        b_2.put("\u04EE", "\u04EF");
        b_2.put("\u04F0", "\u04F1");
        b_2.put("\u04F2", "\u04F3");
        b_2.put("\u04F4", "\u04F5");
        b_2.put("\u04F8", "\u04F9");
        b_2.put("\u0500", "\u0501");
        b_2.put("\u0502", "\u0503");
        b_2.put("\u0504", "\u0505");
        b_2.put("\u0506", "\u0507");
        b_2.put("\u0508", "\u0509");
        b_2.put("\u050A", "\u050B");
        b_2.put("\u050C", "\u050D");
        b_2.put("\u050E", "\u050F");
        b_2.put("\u0531", "\u0561");
        b_2.put("\u0532", "\u0562");
        b_2.put("\u0533", "\u0563");
        b_2.put("\u0534", "\u0564");
        b_2.put("\u0535", "\u0565");
        b_2.put("\u0536", "\u0566");
        b_2.put("\u0537", "\u0567");
        b_2.put("\u0538", "\u0568");
        b_2.put("\u0539", "\u0569");
        b_2.put("\u053A", "\u056A");
        b_2.put("\u053B", "\u056B");
        b_2.put("\u053C", "\u056C");
        b_2.put("\u053D", "\u056D");
        b_2.put("\u053E", "\u056E");
        b_2.put("\u053F", "\u056F");
        b_2.put("\u0540", "\u0570");
        b_2.put("\u0541", "\u0571");
        b_2.put("\u0542", "\u0572");
        b_2.put("\u0543", "\u0573");
        b_2.put("\u0544", "\u0574");
        b_2.put("\u0545", "\u0575");
        b_2.put("\u0546", "\u0576");
        b_2.put("\u0547", "\u0577");
        b_2.put("\u0548", "\u0578");
        b_2.put("\u0549", "\u0579");
        b_2.put("\u054A", "\u057A");
        b_2.put("\u054B", "\u057B");
        b_2.put("\u054C", "\u057C");
        b_2.put("\u054D", "\u057D");
        b_2.put("\u054E", "\u057E");
        b_2.put("\u054F", "\u057F");
        b_2.put("\u0550", "\u0580");
        b_2.put("\u0551", "\u0581");
        b_2.put("\u0552", "\u0582");
        b_2.put("\u0553", "\u0583");
        b_2.put("\u0554", "\u0584");
        b_2.put("\u0555", "\u0585");
        b_2.put("\u0556", "\u0586");
        b_2.put("\u0587", "\u0565\u0582");
        b_2.put("\u1E00", "\u1E01");
        b_2.put("\u1E02", "\u1E03");
        b_2.put("\u1E04", "\u1E05");
        b_2.put("\u1E06", "\u1E07");
        b_2.put("\u1E08", "\u1E09");
        b_2.put("\u1E0A", "\u1E0B");
        b_2.put("\u1E0C", "\u1E0D");
        b_2.put("\u1E0E", "\u1E0F");
        b_2.put("\u1E10", "\u1E11");
        b_2.put("\u1E12", "\u1E13");
        b_2.put("\u1E14", "\u1E15");
        b_2.put("\u1E16", "\u1E17");
        b_2.put("\u1E18", "\u1E19");
        b_2.put("\u1E1A", "\u1E1B");
        b_2.put("\u1E1C", "\u1E1D");
        b_2.put("\u1E1E", "\u1E1F");
        b_2.put("\u1E20", "\u1E21");
        b_2.put("\u1E22", "\u1E23");
        b_2.put("\u1E24", "\u1E25");
        b_2.put("\u1E26", "\u1E27");
        b_2.put("\u1E28", "\u1E29");
        b_2.put("\u1E2A", "\u1E2B");
        b_2.put("\u1E2C", "\u1E2D");
        b_2.put("\u1E2E", "\u1E2F");
        b_2.put("\u1E30", "\u1E31");
        b_2.put("\u1E32", "\u1E33");
        b_2.put("\u1E34", "\u1E35");
        b_2.put("\u1E36", "\u1E37");
        b_2.put("\u1E38", "\u1E39");
        b_2.put("\u1E3A", "\u1E3B");
        b_2.put("\u1E3C", "\u1E3D");
        b_2.put("\u1E3E", "\u1E3F");
        b_2.put("\u1E40", "\u1E41");
        b_2.put("\u1E42", "\u1E43");
        b_2.put("\u1E44", "\u1E45");
        b_2.put("\u1E46", "\u1E47");
        b_2.put("\u1E48", "\u1E49");
        b_2.put("\u1E4A", "\u1E4B");
        b_2.put("\u1E4C", "\u1E4D");
        b_2.put("\u1E4E", "\u1E4F");
        b_2.put("\u1E50", "\u1E51");
        b_2.put("\u1E52", "\u1E53");
        b_2.put("\u1E54", "\u1E55");
        b_2.put("\u1E56", "\u1E57");
        b_2.put("\u1E58", "\u1E59");
        b_2.put("\u1E5A", "\u1E5B");
        b_2.put("\u1E5C", "\u1E5D");
        b_2.put("\u1E5E", "\u1E5F");
        b_2.put("\u1E60", "\u1E61");
        b_2.put("\u1E62", "\u1E63");
        b_2.put("\u1E64", "\u1E65");
        b_2.put("\u1E66", "\u1E67");
        b_2.put("\u1E68", "\u1E69");
        b_2.put("\u1E6A", "\u1E6B");
        b_2.put("\u1E6C", "\u1E6D");
        b_2.put("\u1E6E", "\u1E6F");
        b_2.put("\u1E70", "\u1E71");
        b_2.put("\u1E72", "\u1E73");
        b_2.put("\u1E74", "\u1E75");
        b_2.put("\u1E76", "\u1E77");
        b_2.put("\u1E78", "\u1E79");
        b_2.put("\u1E7A", "\u1E7B");
        b_2.put("\u1E7C", "\u1E7D");
        b_2.put("\u1E7E", "\u1E7F");
        b_2.put("\u1E80", "\u1E81");
        b_2.put("\u1E82", "\u1E83");
        b_2.put("\u1E84", "\u1E85");
        b_2.put("\u1E86", "\u1E87");
        b_2.put("\u1E88", "\u1E89");
        b_2.put("\u1E8A", "\u1E8B");
        b_2.put("\u1E8C", "\u1E8D");
        b_2.put("\u1E8E", "\u1E8F");
        b_2.put("\u1E90", "\u1E91");
        b_2.put("\u1E92", "\u1E93");
        b_2.put("\u1E94", "\u1E95");
        b_2.put("\u1E96", "\u0068\u0331");
        b_2.put("\u1E97", "\u0074\u0308");
        b_2.put("\u1E98", "\u0077\u030A");
        b_2.put("\u1E99", "\u0079\u030A");
        b_2.put("\u1E9A", "\u0061\u02BE");
        b_2.put("\u1E9B", "\u1E61");
        b_2.put("\u1EA0", "\u1EA1");
        b_2.put("\u1EA2", "\u1EA3");
        b_2.put("\u1EA4", "\u1EA5");
        b_2.put("\u1EA6", "\u1EA7");
        b_2.put("\u1EA8", "\u1EA9");
        b_2.put("\u1EAA", "\u1EAB");
        b_2.put("\u1EAC", "\u1EAD");
        b_2.put("\u1EAE", "\u1EAF");
        b_2.put("\u1EB0", "\u1EB1");
        b_2.put("\u1EB2", "\u1EB3");
        b_2.put("\u1EB4", "\u1EB5");
        b_2.put("\u1EB6", "\u1EB7");
        b_2.put("\u1EB8", "\u1EB9");
        b_2.put("\u1EBA", "\u1EBB");
        b_2.put("\u1EBC", "\u1EBD");
        b_2.put("\u1EBE", "\u1EBF");
        b_2.put("\u1EC0", "\u1EC1");
        b_2.put("\u1EC2", "\u1EC3");
        b_2.put("\u1EC4", "\u1EC5");
        b_2.put("\u1EC6", "\u1EC7");
        b_2.put("\u1EC8", "\u1EC9");
        b_2.put("\u1ECA", "\u1ECB");
        b_2.put("\u1ECC", "\u1ECD");
        b_2.put("\u1ECE", "\u1ECF");
        b_2.put("\u1ED0", "\u1ED1");
        b_2.put("\u1ED2", "\u1ED3");
        b_2.put("\u1ED4", "\u1ED5");
        b_2.put("\u1ED6", "\u1ED7");
        b_2.put("\u1ED8", "\u1ED9");
        b_2.put("\u1EDA", "\u1EDB");
        b_2.put("\u1EDC", "\u1EDD");
        b_2.put("\u1EDE", "\u1EDF");
        b_2.put("\u1EE0", "\u1EE1");
        b_2.put("\u1EE2", "\u1EE3");
        b_2.put("\u1EE4", "\u1EE5");
        b_2.put("\u1EE6", "\u1EE7");
        b_2.put("\u1EE8", "\u1EE9");
        b_2.put("\u1EEA", "\u1EEB");
        b_2.put("\u1EEC", "\u1EED");
        b_2.put("\u1EEE", "\u1EEF");
        b_2.put("\u1EF0", "\u1EF1");
        b_2.put("\u1EF2", "\u1EF3");
        b_2.put("\u1EF4", "\u1EF5");
        b_2.put("\u1EF6", "\u1EF7");
        b_2.put("\u1EF8", "\u1EF9");
        b_2.put("\u1F08", "\u1F00");
        b_2.put("\u1F09", "\u1F01");
        b_2.put("\u1F0A", "\u1F02");
        b_2.put("\u1F0B", "\u1F03");
        b_2.put("\u1F0C", "\u1F04");
        b_2.put("\u1F0D", "\u1F05");
        b_2.put("\u1F0E", "\u1F06");
        b_2.put("\u1F0F", "\u1F07");
        b_2.put("\u1F18", "\u1F10");
        b_2.put("\u1F19", "\u1F11");
        b_2.put("\u1F1A", "\u1F12");
        b_2.put("\u1F1B", "\u1F13");
        b_2.put("\u1F1C", "\u1F14");
        b_2.put("\u1F1D", "\u1F15");
        b_2.put("\u1F28", "\u1F20");
        b_2.put("\u1F29", "\u1F21");
        b_2.put("\u1F2A", "\u1F22");
        b_2.put("\u1F2B", "\u1F23");
        b_2.put("\u1F2C", "\u1F24");
        b_2.put("\u1F2D", "\u1F25");
        b_2.put("\u1F2E", "\u1F26");
        b_2.put("\u1F2F", "\u1F27");
        b_2.put("\u1F38", "\u1F30");
        b_2.put("\u1F39", "\u1F31");
        b_2.put("\u1F3A", "\u1F32");
        b_2.put("\u1F3B", "\u1F33");
        b_2.put("\u1F3C", "\u1F34");
        b_2.put("\u1F3D", "\u1F35");
        b_2.put("\u1F3E", "\u1F36");
        b_2.put("\u1F3F", "\u1F37");
        b_2.put("\u1F48", "\u1F40");
        b_2.put("\u1F49", "\u1F41");
        b_2.put("\u1F4A", "\u1F42");
        b_2.put("\u1F4B", "\u1F43");
        b_2.put("\u1F4C", "\u1F44");
        b_2.put("\u1F4D", "\u1F45");
        b_2.put("\u1F50", "\u03C5\u0313");
        b_2.put("\u1F52", "\u03C5\u0313\u0300");
        b_2.put("\u1F54", "\u03C5\u0313\u0301");
        b_2.put("\u1F56", "\u03C5\u0313\u0342");
        b_2.put("\u1F59", "\u1F51");
        b_2.put("\u1F5B", "\u1F53");
        b_2.put("\u1F5D", "\u1F55");
        b_2.put("\u1F5F", "\u1F57");
        b_2.put("\u1F68", "\u1F60");
        b_2.put("\u1F69", "\u1F61");
        b_2.put("\u1F6A", "\u1F62");
        b_2.put("\u1F6B", "\u1F63");
        b_2.put("\u1F6C", "\u1F64");
        b_2.put("\u1F6D", "\u1F65");
        b_2.put("\u1F6E", "\u1F66");
        b_2.put("\u1F6F", "\u1F67");
        b_2.put("\u1F80", "\u1F00\u03B9");
        b_2.put("\u1F81", "\u1F01\u03B9");
        b_2.put("\u1F82", "\u1F02\u03B9");
        b_2.put("\u1F83", "\u1F03\u03B9");
        b_2.put("\u1F84", "\u1F04\u03B9");
        b_2.put("\u1F85", "\u1F05\u03B9");
        b_2.put("\u1F86", "\u1F06\u03B9");
        b_2.put("\u1F87", "\u1F07\u03B9");
        b_2.put("\u1F88", "\u1F00\u03B9");
        b_2.put("\u1F89", "\u1F01\u03B9");
        b_2.put("\u1F8A", "\u1F02\u03B9");
        b_2.put("\u1F8B", "\u1F03\u03B9");
        b_2.put("\u1F8C", "\u1F04\u03B9");
        b_2.put("\u1F8D", "\u1F05\u03B9");
        b_2.put("\u1F8E", "\u1F06\u03B9");
        b_2.put("\u1F8F", "\u1F07\u03B9");
        b_2.put("\u1F90", "\u1F20\u03B9");
        b_2.put("\u1F91", "\u1F21\u03B9");
        b_2.put("\u1F92", "\u1F22\u03B9");
        b_2.put("\u1F93", "\u1F23\u03B9");
        b_2.put("\u1F94", "\u1F24\u03B9");
        b_2.put("\u1F95", "\u1F25\u03B9");
        b_2.put("\u1F96", "\u1F26\u03B9");
        b_2.put("\u1F97", "\u1F27\u03B9");
        b_2.put("\u1F98", "\u1F20\u03B9");
        b_2.put("\u1F99", "\u1F21\u03B9");
        b_2.put("\u1F9A", "\u1F22\u03B9");
        b_2.put("\u1F9B", "\u1F23\u03B9");
        b_2.put("\u1F9C", "\u1F24\u03B9");
        b_2.put("\u1F9D", "\u1F25\u03B9");
        b_2.put("\u1F9E", "\u1F26\u03B9");
        b_2.put("\u1F9F", "\u1F27\u03B9");
        b_2.put("\u1FA0", "\u1F60\u03B9");
        b_2.put("\u1FA1", "\u1F61\u03B9");
        b_2.put("\u1FA2", "\u1F62\u03B9");
        b_2.put("\u1FA3", "\u1F63\u03B9");
        b_2.put("\u1FA4", "\u1F64\u03B9");
        b_2.put("\u1FA5", "\u1F65\u03B9");
        b_2.put("\u1FA6", "\u1F66\u03B9");
        b_2.put("\u1FA7", "\u1F67\u03B9");
        b_2.put("\u1FA8", "\u1F60\u03B9");
        b_2.put("\u1FA9", "\u1F61\u03B9");
        b_2.put("\u1FAA", "\u1F62\u03B9");
        b_2.put("\u1FAB", "\u1F63\u03B9");
        b_2.put("\u1FAC", "\u1F64\u03B9");
        b_2.put("\u1FAD", "\u1F65\u03B9");
        b_2.put("\u1FAE", "\u1F66\u03B9");
        b_2.put("\u1FAF", "\u1F67\u03B9");
        b_2.put("\u1FB2", "\u1F70\u03B9");
        b_2.put("\u1FB3", "\u03B1\u03B9");
        b_2.put("\u1FB4", "\u03AC\u03B9");
        b_2.put("\u1FB6", "\u03B1\u0342");
        b_2.put("\u1FB7", "\u03B1\u0342\u03B9");
        b_2.put("\u1FB8", "\u1FB0");
        b_2.put("\u1FB9", "\u1FB1");
        b_2.put("\u1FBA", "\u1F70");
        b_2.put("\u1FBB", "\u1F71");
        b_2.put("\u1FBC", "\u03B1\u03B9");
        b_2.put("\u1FBE", "\u03B9");
        b_2.put("\u1FC2", "\u1F74\u03B9");
        b_2.put("\u1FC3", "\u03B7\u03B9");
        b_2.put("\u1FC4", "\u03AE\u03B9");
        b_2.put("\u1FC6", "\u03B7\u0342");
        b_2.put("\u1FC7", "\u03B7\u0342\u03B9");
        b_2.put("\u1FC8", "\u1F72");
        b_2.put("\u1FC9", "\u1F73");
        b_2.put("\u1FCA", "\u1F74");
        b_2.put("\u1FCB", "\u1F75");
        b_2.put("\u1FCC", "\u03B7\u03B9");
        b_2.put("\u1FD2", "\u03B9\u0308\u0300");
        b_2.put("\u1FD3", "\u03B9\u0308\u0301");
        b_2.put("\u1FD6", "\u03B9\u0342");
        b_2.put("\u1FD7", "\u03B9\u0308\u0342");
        b_2.put("\u1FD8", "\u1FD0");
        b_2.put("\u1FD9", "\u1FD1");
        b_2.put("\u1FDA", "\u1F76");
        b_2.put("\u1FDB", "\u1F77");
        b_2.put("\u1FE2", "\u03C5\u0308\u0300");
        b_2.put("\u1FE3", "\u03C5\u0308\u0301");
        b_2.put("\u1FE4", "\u03C1\u0313");
        b_2.put("\u1FE6", "\u03C5\u0342");
        b_2.put("\u1FE7", "\u03C5\u0308\u0342");
        b_2.put("\u1FE8", "\u1FE0");
        b_2.put("\u1FE9", "\u1FE1");
        b_2.put("\u1FEA", "\u1F7A");
        b_2.put("\u1FEB", "\u1F7B");
        b_2.put("\u1FEC", "\u1FE5");
        b_2.put("\u1FF2", "\u1F7C\u03B9");
        b_2.put("\u1FF3", "\u03C9\u03B9");
        b_2.put("\u1FF4", "\u03CE\u03B9");
        b_2.put("\u1FF6", "\u03C9\u0342");
        b_2.put("\u1FF7", "\u03C9\u0342\u03B9");
        b_2.put("\u1FF8", "\u1F78");
        b_2.put("\u1FF9", "\u1F79");
        b_2.put("\u1FFA", "\u1F7C");
        b_2.put("\u1FFB", "\u1F7D");
        b_2.put("\u1FFC", "\u03C9\u03B9");
        b_2.put("\u20A8", "\u0072\u0073");
        b_2.put("\u2102", "\u0063");
        b_2.put("\u2103", "\u00B0\u0063");
        b_2.put("\u2107", "\u025B");
        b_2.put("\u2109", "\u00B0\u0066");
        b_2.put("\u210B", "\u0068");
        b_2.put("\u210C", "\u0068");
        b_2.put("\u210D", "\u0068");
        b_2.put("\u2110", "\u0069");
        b_2.put("\u2111", "\u0069");
        b_2.put("\u2112", "\u006C");
        b_2.put("\u2115", "\u006E");
        b_2.put("\u2116", "\u006E\u006F");
        b_2.put("\u2119", "\u0070");
        b_2.put("\u211A", "\u0071");
        b_2.put("\u211B", "\u0072");
        b_2.put("\u211C", "\u0072");
        b_2.put("\u211D", "\u0072");
        b_2.put("\u2120", "\u0073\u006D");
        b_2.put("\u2121", "\u0074\u0065\u006C");
        b_2.put("\u2122", "\u0074\u006D");
        b_2.put("\u2124", "\u007A");
        b_2.put("\u2126", "\u03C9");
        b_2.put("\u2128", "\u007A");
        b_2.put("\u212A", "\u006B");
        b_2.put("\u212B", "\u00E5");
        b_2.put("\u212C", "\u0062");
        b_2.put("\u212D", "\u0063");
        b_2.put("\u2130", "\u0065");
        b_2.put("\u2131", "\u0066");
        b_2.put("\u2133", "\u006D");
        b_2.put("\u213E", "\u03B3");
        b_2.put("\u213F", "\u03C0");
        b_2.put("\u2145", "\u0064");
        b_2.put("\u2160", "\u2170");
        b_2.put("\u2161", "\u2171");
        b_2.put("\u2162", "\u2172");
        b_2.put("\u2163", "\u2173");
        b_2.put("\u2164", "\u2174");
        b_2.put("\u2165", "\u2175");
        b_2.put("\u2166", "\u2176");
        b_2.put("\u2167", "\u2177");
        b_2.put("\u2168", "\u2178");
        b_2.put("\u2169", "\u2179");
        b_2.put("\u216A", "\u217A");
        b_2.put("\u216B", "\u217B");
        b_2.put("\u216C", "\u217C");
        b_2.put("\u216D", "\u217D");
        b_2.put("\u216E", "\u217E");
        b_2.put("\u216F", "\u217F");
        b_2.put("\u24B6", "\u24D0");
        b_2.put("\u24B7", "\u24D1");
        b_2.put("\u24B8", "\u24D2");
        b_2.put("\u24B9", "\u24D3");
        b_2.put("\u24BA", "\u24D4");
        b_2.put("\u24BB", "\u24D5");
        b_2.put("\u24BC", "\u24D6");
        b_2.put("\u24BD", "\u24D7");
        b_2.put("\u24BE", "\u24D8");
        b_2.put("\u24BF", "\u24D9");
        b_2.put("\u24C0", "\u24DA");
        b_2.put("\u24C1", "\u24DB");
        b_2.put("\u24C2", "\u24DC");
        b_2.put("\u24C3", "\u24DD");
        b_2.put("\u24C4", "\u24DE");
        b_2.put("\u24C5", "\u24DF");
        b_2.put("\u24C6", "\u24E0");
        b_2.put("\u24C7", "\u24E1");
        b_2.put("\u24C8", "\u24E2");
        b_2.put("\u24C9", "\u24E3");
        b_2.put("\u24CA", "\u24E4");
        b_2.put("\u24CB", "\u24E5");
        b_2.put("\u24CC", "\u24E6");
        b_2.put("\u24CD", "\u24E7");
        b_2.put("\u24CE", "\u24E8");
        b_2.put("\u24CF", "\u24E9");
        b_2.put("\u3371", "\u0068\u0070\u0061");
        b_2.put("\u3373", "\u0061\u0075");
        b_2.put("\u3375", "\u006F\u0076");
        b_2.put("\u3380", "\u0070\u0061");
        b_2.put("\u3381", "\u006E\u0061");
        b_2.put("\u3382", "\u03BC\u0061");
        b_2.put("\u3383", "\u006D\u0061");
        b_2.put("\u3384", "\u006B\u0061");
        b_2.put("\u3385", "\u006B\u0062");
        b_2.put("\u3386", "\u006D\u0062");
        b_2.put("\u3387", "\u0067\u0062");
        b_2.put("\u338A", "\u0070\u0066");
        b_2.put("\u338B", "\u006E\u0066");
        b_2.put("\u338C", "\u03BC\u0066");
        b_2.put("\u3390", "\u0068\u007A");
        b_2.put("\u3391", "\u006B\u0068\u007A");
        b_2.put("\u3392", "\u006D\u0068\u007A");
        b_2.put("\u3393", "\u0067\u0068\u007A");
        b_2.put("\u3394", "\u0074\u0068\u007A");
        b_2.put("\u33A9", "\u0070\u0061");
        b_2.put("\u33AA", "\u006B\u0070\u0061");
        b_2.put("\u33AB", "\u006D\u0070\u0061");
        b_2.put("\u33AC", "\u0067\u0070\u0061");
        b_2.put("\u33B4", "\u0070\u0076");
        b_2.put("\u33B5", "\u006E\u0076");
        b_2.put("\u33B6", "\u03BC\u0076");
        b_2.put("\u33B7", "\u006D\u0076");
        b_2.put("\u33B8", "\u006B\u0076");
        b_2.put("\u33B9", "\u006D\u0076");
        b_2.put("\u33BA", "\u0070\u0077");
        b_2.put("\u33BB", "\u006E\u0077");
        b_2.put("\u33BC", "\u03BC\u0077");
        b_2.put("\u33BD", "\u006D\u0077");
        b_2.put("\u33BE", "\u006B\u0077");
        b_2.put("\u33BF", "\u006D\u0077");
        b_2.put("\u33C0", "\u006B\u03C9");
        b_2.put("\u33C1", "\u006D\u03C9");
        b_2.put("\u33C3", "\u0062\u0071");
        b_2.put("\u33C6", "\u0063\u2215\u006B\u0067");
        b_2.put("\u33C7", "\u0063\u006F\u002E");
        b_2.put("\u33C8", "\u0064\u0062");
        b_2.put("\u33C9", "\u0067\u0079");
        b_2.put("\u33CB", "\u0068\u0070");
        b_2.put("\u33CD", "\u006B\u006B");
        b_2.put("\u33CE", "\u006B\u006D");
        b_2.put("\u33D7", "\u0070\u0068");
        b_2.put("\u33D9", "\u0070\u0070\u006D");
        b_2.put("\u33DA", "\u0070\u0072");
        b_2.put("\u33DC", "\u0073\u0076");
        b_2.put("\u33DD", "\u0077\u0062");
        b_2.put("\uFB00", "\u0066\u0066");
        b_2.put("\uFB01", "\u0066\u0069");
        b_2.put("\uFB02", "\u0066\u006C");
        b_2.put("\uFB03", "\u0066\u0066\u0069");
        b_2.put("\uFB04", "\u0066\u0066\u006C");
        b_2.put("\uFB05", "\u0073\u0074");
        b_2.put("\uFB06", "\u0073\u0074");
        b_2.put("\uFB13", "\u0574\u0576");
        b_2.put("\uFB14", "\u0574\u0565");
        b_2.put("\uFB15", "\u0574\u056B");
        b_2.put("\uFB16", "\u057E\u0576");
        b_2.put("\uFB17", "\u0574\u056D");
        b_2.put("\uFF21", "\uFF41");
        b_2.put("\uFF22", "\uFF42");
        b_2.put("\uFF23", "\uFF43");
        b_2.put("\uFF24", "\uFF44");
        b_2.put("\uFF25", "\uFF45");
        b_2.put("\uFF26", "\uFF46");
        b_2.put("\uFF27", "\uFF47");
        b_2.put("\uFF28", "\uFF48");
        b_2.put("\uFF29", "\uFF49");
        b_2.put("\uFF2A", "\uFF4A");
        b_2.put("\uFF2B", "\uFF4B");
        b_2.put("\uFF2C", "\uFF4C");
        b_2.put("\uFF2D", "\uFF4D");
        b_2.put("\uFF2E", "\uFF4E");
        b_2.put("\uFF2F", "\uFF4F");
        b_2.put("\uFF30", "\uFF50");
        b_2.put("\uFF31", "\uFF51");
        b_2.put("\uFF32", "\uFF52");
        b_2.put("\uFF33", "\uFF53");
        b_2.put("\uFF34", "\uFF54");
        b_2.put("\uFF35", "\uFF55");
        b_2.put("\uFF36", "\uFF56");
        b_2.put("\uFF37", "\uFF57");
        b_2.put("\uFF38", "\uFF58");
        b_2.put("\uFF39", "\uFF59");
        b_2.put("\uFF3A", "\uFF5A");
        b_2.put(codePointToString(0x10400), codePointToString(0x10428));
        b_2.put(codePointToString(0x10401), codePointToString(0x10429));
        b_2.put(codePointToString(0x10402), codePointToString(0x1042A));
        b_2.put(codePointToString(0x10403), codePointToString(0x1042B));
        b_2.put(codePointToString(0x10404), codePointToString(0x1042C));
        b_2.put(codePointToString(0x10405), codePointToString(0x1042D));
        b_2.put(codePointToString(0x10406), codePointToString(0x1042E));
        b_2.put(codePointToString(0x10407), codePointToString(0x1042F));
        b_2.put(codePointToString(0x10408), codePointToString(0x10430));
        b_2.put(codePointToString(0x10409), codePointToString(0x10431));
        b_2.put(codePointToString(0x1040A), codePointToString(0x10432));
        b_2.put(codePointToString(0x1040B), codePointToString(0x10433));
        b_2.put(codePointToString(0x1040C), codePointToString(0x10434));
        b_2.put(codePointToString(0x1040D), codePointToString(0x10435));
        b_2.put(codePointToString(0x1040E), codePointToString(0x10436));
        b_2.put(codePointToString(0x1040F), codePointToString(0x10437));
        b_2.put(codePointToString(0x10410), codePointToString(0x10438));
        b_2.put(codePointToString(0x10411), codePointToString(0x10439));
        b_2.put(codePointToString(0x10412), codePointToString(0x1043A));
        b_2.put(codePointToString(0x10413), codePointToString(0x1043B));
        b_2.put(codePointToString(0x10414), codePointToString(0x1043C));
        b_2.put(codePointToString(0x10415), codePointToString(0x1043D));
        b_2.put(codePointToString(0x10416), codePointToString(0x1043E));
        b_2.put(codePointToString(0x10417), codePointToString(0x1043F));
        b_2.put(codePointToString(0x10418), codePointToString(0x10440));
        b_2.put(codePointToString(0x10419), codePointToString(0x10441));
        b_2.put(codePointToString(0x1041A), codePointToString(0x10442));
        b_2.put(codePointToString(0x1041B), codePointToString(0x10443));
        b_2.put(codePointToString(0x1041C), codePointToString(0x10444));
        b_2.put(codePointToString(0x1041D), codePointToString(0x10445));
        b_2.put(codePointToString(0x1041E), codePointToString(0x10446));
        b_2.put(codePointToString(0x1041F), codePointToString(0x10447));
        b_2.put(codePointToString(0x10420), codePointToString(0x10448));
        b_2.put(codePointToString(0x10421), codePointToString(0x10449));
        b_2.put(codePointToString(0x10422), codePointToString(0x1044A));
        b_2.put(codePointToString(0x10423), codePointToString(0x1044B));
        b_2.put(codePointToString(0x10424), codePointToString(0x1044C));
        b_2.put(codePointToString(0x10425), codePointToString(0x1044D));
        b_2.put(codePointToString(0x1D400), "\u0061");
        b_2.put(codePointToString(0x1D401), "\u0062");
        b_2.put(codePointToString(0x1D402), "\u0063");
        b_2.put(codePointToString(0x1D403), "\u0064");
        b_2.put(codePointToString(0x1D404), "\u0065");
        b_2.put(codePointToString(0x1D405), "\u0066");
        b_2.put(codePointToString(0x1D406), "\u0067");
        b_2.put(codePointToString(0x1D407), "\u0068");
        b_2.put(codePointToString(0x1D408), "\u0069");
        b_2.put(codePointToString(0x1D409), "\u006A");
        b_2.put(codePointToString(0x1D40A), "\u006B");
        b_2.put(codePointToString(0x1D40B), "\u006C");
        b_2.put(codePointToString(0x1D40C), "\u006D");
        b_2.put(codePointToString(0x1D40D), "\u006E");
        b_2.put(codePointToString(0x1D40E), "\u006F");
        b_2.put(codePointToString(0x1D40F), "\u0070");
        b_2.put(codePointToString(0x1D410), "\u0071");
        b_2.put(codePointToString(0x1D411), "\u0072");
        b_2.put(codePointToString(0x1D412), "\u0073");
        b_2.put(codePointToString(0x1D413), "\u0074");
        b_2.put(codePointToString(0x1D414), "\u0075");
        b_2.put(codePointToString(0x1D415), "\u0076");
        b_2.put(codePointToString(0x1D416), "\u0077");
        b_2.put(codePointToString(0x1D417), "\u0078");
        b_2.put(codePointToString(0x1D418), "\u0079");
        b_2.put(codePointToString(0x1D419), "\u007A");
        b_2.put(codePointToString(0x1D434), "\u0061");
        b_2.put(codePointToString(0x1D435), "\u0062");
        b_2.put(codePointToString(0x1D436), "\u0063");
        b_2.put(codePointToString(0x1D437), "\u0064");
        b_2.put(codePointToString(0x1D438), "\u0065");
        b_2.put(codePointToString(0x1D439), "\u0066");
        b_2.put(codePointToString(0x1D43A), "\u0067");
        b_2.put(codePointToString(0x1D43B), "\u0068");
        b_2.put(codePointToString(0x1D43C), "\u0069");
        b_2.put(codePointToString(0x1D43D), "\u006A");
        b_2.put(codePointToString(0x1D43E), "\u006B");
        b_2.put(codePointToString(0x1D43F), "\u006C");
        b_2.put(codePointToString(0x1D440), "\u006D");
        b_2.put(codePointToString(0x1D441), "\u006E");
        b_2.put(codePointToString(0x1D442), "\u006F");
        b_2.put(codePointToString(0x1D443), "\u0070");
        b_2.put(codePointToString(0x1D444), "\u0071");
        b_2.put(codePointToString(0x1D445), "\u0072");
        b_2.put(codePointToString(0x1D446), "\u0073");
        b_2.put(codePointToString(0x1D447), "\u0074");
        b_2.put(codePointToString(0x1D448), "\u0075");
        b_2.put(codePointToString(0x1D449), "\u0076");
        b_2.put(codePointToString(0x1D44A), "\u0077");
        b_2.put(codePointToString(0x1D44B), "\u0078");
        b_2.put(codePointToString(0x1D44C), "\u0079");
        b_2.put(codePointToString(0x1D44D), "\u007A");
        b_2.put(codePointToString(0x1D468), "\u0061");
        b_2.put(codePointToString(0x1D469), "\u0062");
        b_2.put(codePointToString(0x1D46A), "\u0063");
        b_2.put(codePointToString(0x1D46B), "\u0064");
        b_2.put(codePointToString(0x1D46C), "\u0065");
        b_2.put(codePointToString(0x1D46D), "\u0066");
        b_2.put(codePointToString(0x1D46E), "\u0067");
        b_2.put(codePointToString(0x1D46F), "\u0068");
        b_2.put(codePointToString(0x1D470), "\u0069");
        b_2.put(codePointToString(0x1D471), "\u006A");
        b_2.put(codePointToString(0x1D472), "\u006B");
        b_2.put(codePointToString(0x1D473), "\u006C");
        b_2.put(codePointToString(0x1D474), "\u006D");
        b_2.put(codePointToString(0x1D475), "\u006E");
        b_2.put(codePointToString(0x1D476), "\u006F");
        b_2.put(codePointToString(0x1D477), "\u0070");
        b_2.put(codePointToString(0x1D478), "\u0071");
        b_2.put(codePointToString(0x1D479), "\u0072");
        b_2.put(codePointToString(0x1D47A), "\u0073");
        b_2.put(codePointToString(0x1D47B), "\u0074");
        b_2.put(codePointToString(0x1D47C), "\u0075");
        b_2.put(codePointToString(0x1D47D), "\u0076");
        b_2.put(codePointToString(0x1D47E), "\u0077");
        b_2.put(codePointToString(0x1D47F), "\u0078");
        b_2.put(codePointToString(0x1D480), "\u0079");
        b_2.put(codePointToString(0x1D481), "\u007A");
        b_2.put(codePointToString(0x1D49C), "\u0061");
        b_2.put(codePointToString(0x1D49E), "\u0063");
        b_2.put(codePointToString(0x1D49F), "\u0064");
        b_2.put(codePointToString(0x1D4A2), "\u0067");
        b_2.put(codePointToString(0x1D4A5), "\u006A");
        b_2.put(codePointToString(0x1D4A6), "\u006B");
        b_2.put(codePointToString(0x1D4A9), "\u006E");
        b_2.put(codePointToString(0x1D4AA), "\u006F");
        b_2.put(codePointToString(0x1D4AB), "\u0070");
        b_2.put(codePointToString(0x1D4AC), "\u0071");
        b_2.put(codePointToString(0x1D4AE), "\u0073");
        b_2.put(codePointToString(0x1D4AF), "\u0074");
        b_2.put(codePointToString(0x1D4B0), "\u0075");
        b_2.put(codePointToString(0x1D4B1), "\u0076");
        b_2.put(codePointToString(0x1D4B2), "\u0077");
        b_2.put(codePointToString(0x1D4B3), "\u0078");
        b_2.put(codePointToString(0x1D4B4), "\u0079");
        b_2.put(codePointToString(0x1D4B5), "\u007A");
        b_2.put(codePointToString(0x1D4D0), "\u0061");
        b_2.put(codePointToString(0x1D4D1), "\u0062");
        b_2.put(codePointToString(0x1D4D2), "\u0063");
        b_2.put(codePointToString(0x1D4D3), "\u0064");
        b_2.put(codePointToString(0x1D4D4), "\u0065");
        b_2.put(codePointToString(0x1D4D5), "\u0066");
        b_2.put(codePointToString(0x1D4D6), "\u0067");
        b_2.put(codePointToString(0x1D4D7), "\u0068");
        b_2.put(codePointToString(0x1D4D8), "\u0069");
        b_2.put(codePointToString(0x1D4D9), "\u006A");
        b_2.put(codePointToString(0x1D4DA), "\u006B");
        b_2.put(codePointToString(0x1D4DB), "\u006C");
        b_2.put(codePointToString(0x1D4DC), "\u006D");
        b_2.put(codePointToString(0x1D4DD), "\u006E");
        b_2.put(codePointToString(0x1D4DE), "\u006F");
        b_2.put(codePointToString(0x1D4DF), "\u0070");
        b_2.put(codePointToString(0x1D4E0), "\u0071");
        b_2.put(codePointToString(0x1D4E1), "\u0072");
        b_2.put(codePointToString(0x1D4E2), "\u0073");
        b_2.put(codePointToString(0x1D4E3), "\u0074");
        b_2.put(codePointToString(0x1D4E4), "\u0075");
        b_2.put(codePointToString(0x1D4E5), "\u0076");
        b_2.put(codePointToString(0x1D4E6), "\u0077");
        b_2.put(codePointToString(0x1D4E7), "\u0078");
        b_2.put(codePointToString(0x1D4E8), "\u0079");
        b_2.put(codePointToString(0x1D4E9), "\u007A");
        b_2.put(codePointToString(0x1D504), "\u0061");
        b_2.put(codePointToString(0x1D505), "\u0062");
        b_2.put(codePointToString(0x1D507), "\u0064");
        b_2.put(codePointToString(0x1D508), "\u0065");
        b_2.put(codePointToString(0x1D509), "\u0066");
        b_2.put(codePointToString(0x1D50A), "\u0067");
        b_2.put(codePointToString(0x1D50D), "\u006A");
        b_2.put(codePointToString(0x1D50E), "\u006B");
        b_2.put(codePointToString(0x1D50F), "\u006C");
        b_2.put(codePointToString(0x1D510), "\u006D");
        b_2.put(codePointToString(0x1D511), "\u006E");
        b_2.put(codePointToString(0x1D512), "\u006F");
        b_2.put(codePointToString(0x1D513), "\u0070");
        b_2.put(codePointToString(0x1D514), "\u0071");
        b_2.put(codePointToString(0x1D516), "\u0073");
        b_2.put(codePointToString(0x1D517), "\u0074");
        b_2.put(codePointToString(0x1D518), "\u0075");
        b_2.put(codePointToString(0x1D519), "\u0076");
        b_2.put(codePointToString(0x1D51A), "\u0077");
        b_2.put(codePointToString(0x1D51B), "\u0078");
        b_2.put(codePointToString(0x1D51C), "\u0079");
        b_2.put(codePointToString(0x1D538), "\u0061");
        b_2.put(codePointToString(0x1D539), "\u0062");
        b_2.put(codePointToString(0x1D53B), "\u0064");
        b_2.put(codePointToString(0x1D53C), "\u0065");
        b_2.put(codePointToString(0x1D53D), "\u0066");
        b_2.put(codePointToString(0x1D53E), "\u0067");
        b_2.put(codePointToString(0x1D540), "\u0069");
        b_2.put(codePointToString(0x1D541), "\u006A");
        b_2.put(codePointToString(0x1D542), "\u006B");
        b_2.put(codePointToString(0x1D543), "\u006C");
        b_2.put(codePointToString(0x1D544), "\u006D");
        b_2.put(codePointToString(0x1D546), "\u006F");
        b_2.put(codePointToString(0x1D54A), "\u0073");
        b_2.put(codePointToString(0x1D54B), "\u0074");
        b_2.put(codePointToString(0x1D54C), "\u0075");
        b_2.put(codePointToString(0x1D54D), "\u0076");
        b_2.put(codePointToString(0x1D54E), "\u0077");
        b_2.put(codePointToString(0x1D54F), "\u0078");
        b_2.put(codePointToString(0x1D550), "\u0079");
        b_2.put(codePointToString(0x1D56C), "\u0061");
        b_2.put(codePointToString(0x1D56D), "\u0062");
        b_2.put(codePointToString(0x1D56E), "\u0063");
        b_2.put(codePointToString(0x1D56F), "\u0064");
        b_2.put(codePointToString(0x1D570), "\u0065");
        b_2.put(codePointToString(0x1D571), "\u0066");
        b_2.put(codePointToString(0x1D572), "\u0067");
        b_2.put(codePointToString(0x1D573), "\u0068");
        b_2.put(codePointToString(0x1D574), "\u0069");
        b_2.put(codePointToString(0x1D575), "\u006A");
        b_2.put(codePointToString(0x1D576), "\u006B");
        b_2.put(codePointToString(0x1D577), "\u006C");
        b_2.put(codePointToString(0x1D578), "\u006D");
        b_2.put(codePointToString(0x1D579), "\u006E");
        b_2.put(codePointToString(0x1D57A), "\u006F");
        b_2.put(codePointToString(0x1D57B), "\u0070");
        b_2.put(codePointToString(0x1D57C), "\u0071");
        b_2.put(codePointToString(0x1D57D), "\u0072");
        b_2.put(codePointToString(0x1D57E), "\u0073");
        b_2.put(codePointToString(0x1D57F), "\u0074");
        b_2.put(codePointToString(0x1D580), "\u0075");
        b_2.put(codePointToString(0x1D581), "\u0076");
        b_2.put(codePointToString(0x1D582), "\u0077");
        b_2.put(codePointToString(0x1D583), "\u0078");
        b_2.put(codePointToString(0x1D584), "\u0079");
        b_2.put(codePointToString(0x1D585), "\u007A");
        b_2.put(codePointToString(0x1D5A0), "\u0061");
        b_2.put(codePointToString(0x1D5A1), "\u0062");
        b_2.put(codePointToString(0x1D5A2), "\u0063");
        b_2.put(codePointToString(0x1D5A3), "\u0064");
        b_2.put(codePointToString(0x1D5A4), "\u0065");
        b_2.put(codePointToString(0x1D5A5), "\u0066");
        b_2.put(codePointToString(0x1D5A6), "\u0067");
        b_2.put(codePointToString(0x1D5A7), "\u0068");
        b_2.put(codePointToString(0x1D5A8), "\u0069");
        b_2.put(codePointToString(0x1D5A9), "\u006A");
        b_2.put(codePointToString(0x1D5AA), "\u006B");
        b_2.put(codePointToString(0x1D5AB), "\u006C");
        b_2.put(codePointToString(0x1D5AC), "\u006D");
        b_2.put(codePointToString(0x1D5AD), "\u006E");
        b_2.put(codePointToString(0x1D5AE), "\u006F");
        b_2.put(codePointToString(0x1D5AF), "\u0070");
        b_2.put(codePointToString(0x1D5B0), "\u0071");
        b_2.put(codePointToString(0x1D5B1), "\u0072");
        b_2.put(codePointToString(0x1D5B2), "\u0073");
        b_2.put(codePointToString(0x1D5B3), "\u0074");
        b_2.put(codePointToString(0x1D5B4), "\u0075");
        b_2.put(codePointToString(0x1D5B5), "\u0076");
        b_2.put(codePointToString(0x1D5B6), "\u0077");
        b_2.put(codePointToString(0x1D5B7), "\u0078");
        b_2.put(codePointToString(0x1D5B8), "\u0079");
        b_2.put(codePointToString(0x1D5B9), "\u007A");
        b_2.put(codePointToString(0x1D5D4), "\u0061");
        b_2.put(codePointToString(0x1D5D5), "\u0062");
        b_2.put(codePointToString(0x1D5D6), "\u0063");
        b_2.put(codePointToString(0x1D5D7), "\u0064");
        b_2.put(codePointToString(0x1D5D8), "\u0065");
        b_2.put(codePointToString(0x1D5D9), "\u0066");
        b_2.put(codePointToString(0x1D5DA), "\u0067");
        b_2.put(codePointToString(0x1D5DB), "\u0068");
        b_2.put(codePointToString(0x1D5DC), "\u0069");
        b_2.put(codePointToString(0x1D5DD), "\u006A");
        b_2.put(codePointToString(0x1D5DE), "\u006B");
        b_2.put(codePointToString(0x1D5DF), "\u006C");
        b_2.put(codePointToString(0x1D5E0), "\u006D");
        b_2.put(codePointToString(0x1D5E1), "\u006E");
        b_2.put(codePointToString(0x1D5E2), "\u006F");
        b_2.put(codePointToString(0x1D5E3), "\u0070");
        b_2.put(codePointToString(0x1D5E4), "\u0071");
        b_2.put(codePointToString(0x1D5E5), "\u0072");
        b_2.put(codePointToString(0x1D5E6), "\u0073");
        b_2.put(codePointToString(0x1D5E7), "\u0074");
        b_2.put(codePointToString(0x1D5E8), "\u0075");
        b_2.put(codePointToString(0x1D5E9), "\u0076");
        b_2.put(codePointToString(0x1D5EA), "\u0077");
        b_2.put(codePointToString(0x1D5EB), "\u0078");
        b_2.put(codePointToString(0x1D5EC), "\u0079");
        b_2.put(codePointToString(0x1D5ED), "\u007A");
        b_2.put(codePointToString(0x1D608), "\u0061");
        b_2.put(codePointToString(0x1D609), "\u0062");
        b_2.put(codePointToString(0x1D60A), "\u0063");
        b_2.put(codePointToString(0x1D60B), "\u0064");
        b_2.put(codePointToString(0x1D60C), "\u0065");
        b_2.put(codePointToString(0x1D60D), "\u0066");
        b_2.put(codePointToString(0x1D60E), "\u0067");
        b_2.put(codePointToString(0x1D60F), "\u0068");
        b_2.put(codePointToString(0x1D610), "\u0069");
        b_2.put(codePointToString(0x1D611), "\u006A");
        b_2.put(codePointToString(0x1D612), "\u006B");
        b_2.put(codePointToString(0x1D613), "\u006C");
        b_2.put(codePointToString(0x1D614), "\u006D");
        b_2.put(codePointToString(0x1D615), "\u006E");
        b_2.put(codePointToString(0x1D616), "\u006F");
        b_2.put(codePointToString(0x1D617), "\u0070");
        b_2.put(codePointToString(0x1D618), "\u0071");
        b_2.put(codePointToString(0x1D619), "\u0072");
        b_2.put(codePointToString(0x1D61A), "\u0073");
        b_2.put(codePointToString(0x1D61B), "\u0074");
        b_2.put(codePointToString(0x1D61C), "\u0075");
        b_2.put(codePointToString(0x1D61D), "\u0076");
        b_2.put(codePointToString(0x1D61E), "\u0077");
        b_2.put(codePointToString(0x1D61F), "\u0078");
        b_2.put(codePointToString(0x1D620), "\u0079");
        b_2.put(codePointToString(0x1D621), "\u007A");
        b_2.put(codePointToString(0x1D63C), "\u0061");
        b_2.put(codePointToString(0x1D63D), "\u0062");
        b_2.put(codePointToString(0x1D63E), "\u0063");
        b_2.put(codePointToString(0x1D63F), "\u0064");
        b_2.put(codePointToString(0x1D640), "\u0065");
        b_2.put(codePointToString(0x1D641), "\u0066");
        b_2.put(codePointToString(0x1D642), "\u0067");
        b_2.put(codePointToString(0x1D643), "\u0068");
        b_2.put(codePointToString(0x1D644), "\u0069");
        b_2.put(codePointToString(0x1D645), "\u006A");
        b_2.put(codePointToString(0x1D646), "\u006B");
        b_2.put(codePointToString(0x1D647), "\u006C");
        b_2.put(codePointToString(0x1D648), "\u006D");
        b_2.put(codePointToString(0x1D649), "\u006E");
        b_2.put(codePointToString(0x1D64A), "\u006F");
        b_2.put(codePointToString(0x1D64B), "\u0070");
        b_2.put(codePointToString(0x1D64C), "\u0071");
        b_2.put(codePointToString(0x1D64D), "\u0072");
        b_2.put(codePointToString(0x1D64E), "\u0073");
        b_2.put(codePointToString(0x1D64F), "\u0074");
        b_2.put(codePointToString(0x1D650), "\u0075");
        b_2.put(codePointToString(0x1D651), "\u0076");
        b_2.put(codePointToString(0x1D652), "\u0077");
        b_2.put(codePointToString(0x1D653), "\u0078");
        b_2.put(codePointToString(0x1D654), "\u0079");
        b_2.put(codePointToString(0x1D655), "\u007A");
        b_2.put(codePointToString(0x1D670), "\u0061");
        b_2.put(codePointToString(0x1D671), "\u0062");
        b_2.put(codePointToString(0x1D672), "\u0063");
        b_2.put(codePointToString(0x1D673), "\u0064");
        b_2.put(codePointToString(0x1D674), "\u0065");
        b_2.put(codePointToString(0x1D675), "\u0066");
        b_2.put(codePointToString(0x1D676), "\u0067");
        b_2.put(codePointToString(0x1D677), "\u0068");
        b_2.put(codePointToString(0x1D678), "\u0069");
        b_2.put(codePointToString(0x1D679), "\u006A");
        b_2.put(codePointToString(0x1D67A), "\u006B");
        b_2.put(codePointToString(0x1D67B), "\u006C");
        b_2.put(codePointToString(0x1D67C), "\u006D");
        b_2.put(codePointToString(0x1D67D), "\u006E");
        b_2.put(codePointToString(0x1D67E), "\u006F");
        b_2.put(codePointToString(0x1D67F), "\u0070");
        b_2.put(codePointToString(0x1D680), "\u0071");
        b_2.put(codePointToString(0x1D681), "\u0072");
        b_2.put(codePointToString(0x1D682), "\u0073");
        b_2.put(codePointToString(0x1D683), "\u0074");
        b_2.put(codePointToString(0x1D684), "\u0075");
        b_2.put(codePointToString(0x1D685), "\u0076");
        b_2.put(codePointToString(0x1D686), "\u0077");
        b_2.put(codePointToString(0x1D687), "\u0078");
        b_2.put(codePointToString(0x1D688), "\u0079");
        b_2.put(codePointToString(0x1D689), "\u007A");
        b_2.put(codePointToString(0x1D6A8), "\u03B1");
        b_2.put(codePointToString(0x1D6A9), "\u03B2");
        b_2.put(codePointToString(0x1D6AA), "\u03B3");
        b_2.put(codePointToString(0x1D6AB), "\u03B4");
        b_2.put(codePointToString(0x1D6AC), "\u03B5");
        b_2.put(codePointToString(0x1D6AD), "\u03B6");
        b_2.put(codePointToString(0x1D6AE), "\u03B7");
        b_2.put(codePointToString(0x1D6AF), "\u03B8");
        b_2.put(codePointToString(0x1D6B0), "\u03B9");
        b_2.put(codePointToString(0x1D6B1), "\u03BA");
        b_2.put(codePointToString(0x1D6B2), "\u03BB");
        b_2.put(codePointToString(0x1D6B3), "\u03BC");
        b_2.put(codePointToString(0x1D6B4), "\u03BD");
        b_2.put(codePointToString(0x1D6B5), "\u03BE");
        b_2.put(codePointToString(0x1D6B6), "\u03BF");
        b_2.put(codePointToString(0x1D6B7), "\u03C0");
        b_2.put(codePointToString(0x1D6B8), "\u03C1");
        b_2.put(codePointToString(0x1D6B9), "\u03B8");
        b_2.put(codePointToString(0x1D6BA), "\u03C3");
        b_2.put(codePointToString(0x1D6BB), "\u03C4");
        b_2.put(codePointToString(0x1D6BC), "\u03C5");
        b_2.put(codePointToString(0x1D6BD), "\u03C6");
        b_2.put(codePointToString(0x1D6BE), "\u03C7");
        b_2.put(codePointToString(0x1D6BF), "\u03C8");
        b_2.put(codePointToString(0x1D6C0), "\u03C9");
        b_2.put(codePointToString(0x1D6D3), "\u03C3");
        b_2.put(codePointToString(0x1D6E2), "\u03B1");
        b_2.put(codePointToString(0x1D6E3), "\u03B2");
        b_2.put(codePointToString(0x1D6E4), "\u03B3");
        b_2.put(codePointToString(0x1D6E5), "\u03B4");
        b_2.put(codePointToString(0x1D6E6), "\u03B5");
        b_2.put(codePointToString(0x1D6E7), "\u03B6");
        b_2.put(codePointToString(0x1D6E8), "\u03B7");
        b_2.put(codePointToString(0x1D6E9), "\u03B8");
        b_2.put(codePointToString(0x1D6EA), "\u03B9");
        b_2.put(codePointToString(0x1D6EB), "\u03BA");
        b_2.put(codePointToString(0x1D6EC), "\u03BB");
        b_2.put(codePointToString(0x1D6ED), "\u03BC");
        b_2.put(codePointToString(0x1D6EE), "\u03BD");
        b_2.put(codePointToString(0x1D6EF), "\u03BE");
        b_2.put(codePointToString(0x1D6F0), "\u03BF");
        b_2.put(codePointToString(0x1D6F1), "\u03C0");
        b_2.put(codePointToString(0x1D6F2), "\u03C1");
        b_2.put(codePointToString(0x1D6F3), "\u03B8");
        b_2.put(codePointToString(0x1D6F4), "\u03C3");
        b_2.put(codePointToString(0x1D6F5), "\u03C4");
        b_2.put(codePointToString(0x1D6F6), "\u03C5");
        b_2.put(codePointToString(0x1D6F7), "\u03C6");
        b_2.put(codePointToString(0x1D6F8), "\u03C7");
        b_2.put(codePointToString(0x1D6F9), "\u03C8");
        b_2.put(codePointToString(0x1D6FA), "\u03C9");
        b_2.put(codePointToString(0x1D70D), "\u03C3");
        b_2.put(codePointToString(0x1D71C), "\u03B1");
        b_2.put(codePointToString(0x1D71D), "\u03B2");
        b_2.put(codePointToString(0x1D71E), "\u03B3");
        b_2.put(codePointToString(0x1D71F), "\u03B4");
        b_2.put(codePointToString(0x1D720), "\u03B5");
        b_2.put(codePointToString(0x1D721), "\u03B6");
        b_2.put(codePointToString(0x1D722), "\u03B7");
        b_2.put(codePointToString(0x1D723), "\u03B8");
        b_2.put(codePointToString(0x1D724), "\u03B9");
        b_2.put(codePointToString(0x1D725), "\u03BA");
        b_2.put(codePointToString(0x1D726), "\u03BB");
        b_2.put(codePointToString(0x1D727), "\u03BC");
        b_2.put(codePointToString(0x1D728), "\u03BD");
        b_2.put(codePointToString(0x1D729), "\u03BE");
        b_2.put(codePointToString(0x1D72A), "\u03BF");
        b_2.put(codePointToString(0x1D72B), "\u03C0");
        b_2.put(codePointToString(0x1D72C), "\u03C1");
        b_2.put(codePointToString(0x1D72D), "\u03B8");
        b_2.put(codePointToString(0x1D72E), "\u03C3");
        b_2.put(codePointToString(0x1D72F), "\u03C4");
        b_2.put(codePointToString(0x1D730), "\u03C5");
        b_2.put(codePointToString(0x1D731), "\u03C6");
        b_2.put(codePointToString(0x1D732), "\u03C7");
        b_2.put(codePointToString(0x1D733), "\u03C8");
        b_2.put(codePointToString(0x1D734), "\u03C9");
        b_2.put(codePointToString(0x1D747), "\u03C3");
        b_2.put(codePointToString(0x1D756), "\u03B1");
        b_2.put(codePointToString(0x1D757), "\u03B2");
        b_2.put(codePointToString(0x1D758), "\u03B3");
        b_2.put(codePointToString(0x1D759), "\u03B4");
        b_2.put(codePointToString(0x1D75A), "\u03B5");
        b_2.put(codePointToString(0x1D75B), "\u03B6");
        b_2.put(codePointToString(0x1D75C), "\u03B7");
        b_2.put(codePointToString(0x1D75D), "\u03B8");
        b_2.put(codePointToString(0x1D75E), "\u03B9");
        b_2.put(codePointToString(0x1D75F), "\u03BA");
        b_2.put(codePointToString(0x1D760), "\u03BB");
        b_2.put(codePointToString(0x1D761), "\u03BC");
        b_2.put(codePointToString(0x1D762), "\u03BD");
        b_2.put(codePointToString(0x1D763), "\u03BE");
        b_2.put(codePointToString(0x1D764), "\u03BF");
        b_2.put(codePointToString(0x1D765), "\u03C0");
        b_2.put(codePointToString(0x1D766), "\u03C1");
        b_2.put(codePointToString(0x1D767), "\u03B8");
        b_2.put(codePointToString(0x1D768), "\u03C3");
        b_2.put(codePointToString(0x1D769), "\u03C4");
        b_2.put(codePointToString(0x1D76A), "\u03C5");
        b_2.put(codePointToString(0x1D76B), "\u03C6");
        b_2.put(codePointToString(0x1D76C), "\u03C7");
        b_2.put(codePointToString(0x1D76D), "\u03C8");
        b_2.put(codePointToString(0x1D76E), "\u03C9");
        b_2.put(codePointToString(0x1D781), "\u03C3");
        b_2.put(codePointToString(0x1D790), "\u03B1");
        b_2.put(codePointToString(0x1D791), "\u03B2");
        b_2.put(codePointToString(0x1D792), "\u03B3");
        b_2.put(codePointToString(0x1D793), "\u03B4");
        b_2.put(codePointToString(0x1D794), "\u03B5");
        b_2.put(codePointToString(0x1D795), "\u03B6");
        b_2.put(codePointToString(0x1D796), "\u03B7");
        b_2.put(codePointToString(0x1D797), "\u03B8");
        b_2.put(codePointToString(0x1D798), "\u03B9");
        b_2.put(codePointToString(0x1D799), "\u03BA");
        b_2.put(codePointToString(0x1D79A), "\u03BB");
        b_2.put(codePointToString(0x1D79B), "\u03BC");
        b_2.put(codePointToString(0x1D79C), "\u03BD");
        b_2.put(codePointToString(0x1D79D), "\u03BE");
        b_2.put(codePointToString(0x1D79E), "\u03BF");
        b_2.put(codePointToString(0x1D79F), "\u03C0");
        b_2.put(codePointToString(0x1D7A0), "\u03C1");
        b_2.put(codePointToString(0x1D7A1), "\u03B8");
        b_2.put(codePointToString(0x1D7A2), "\u03C3");
        b_2.put(codePointToString(0x1D7A3), "\u03C4");
        b_2.put(codePointToString(0x1D7A4), "\u03C5");
        b_2.put(codePointToString(0x1D7A5), "\u03C6");
        b_2.put(codePointToString(0x1D7A6), "\u03C7");
        b_2.put(codePointToString(0x1D7A7), "\u03C8");
        b_2.put(codePointToString(0x1D7A8), "\u03C9");
        b_2.put(codePointToString(0x1D7BB), "\u03C3");
        return Collections.unmodifiableMap(b_2);
    }

    private static final Map<String, String> build_B_3_MappingForCaseFoldingWithNoNormalization() {
        Map<String, String> b_3 = new HashMap<String, String>();
        b_3.put("\u0041", "\u0061");
        b_3.put("\u0042", "\u0062");
        b_3.put("\u0043", "\u0063");
        b_3.put("\u0044", "\u0064");
        b_3.put("\u0045", "\u0065");
        b_3.put("\u0046", "\u0066");
        b_3.put("\u0047", "\u0067");
        b_3.put("\u0048", "\u0068");
        b_3.put("\u0049", "\u0069");
        b_3.put("\u004A", "\u006A");
        b_3.put("\u004B", "\u006B");
        b_3.put("\u004C", "\u006C");
        b_3.put("\u004D", "\u006D");
        b_3.put("\u004E", "\u006E");
        b_3.put("\u004F", "\u006F");
        b_3.put("\u0050", "\u0070");
        b_3.put("\u0051", "\u0071");
        b_3.put("\u0052", "\u0072");
        b_3.put("\u0053", "\u0073");
        b_3.put("\u0054", "\u0074");
        b_3.put("\u0055", "\u0075");
        b_3.put("\u0056", "\u0076");
        b_3.put("\u0057", "\u0077");
        b_3.put("\u0058", "\u0078");
        b_3.put("\u0059", "\u0079");
        b_3.put("\u005A", "\u007A");
        b_3.put("\u00B5", "\u03BC");
        b_3.put("\u00C0", "\u00E0");
        b_3.put("\u00C1", "\u00E1");
        b_3.put("\u00C2", "\u00E2");
        b_3.put("\u00C3", "\u00E3");
        b_3.put("\u00C4", "\u00E4");
        b_3.put("\u00C5", "\u00E5");
        b_3.put("\u00C6", "\u00E6");
        b_3.put("\u00C7", "\u00E7");
        b_3.put("\u00C8", "\u00E8");
        b_3.put("\u00C9", "\u00E9");
        b_3.put("\u00CA", "\u00EA");
        b_3.put("\u00CB", "\u00EB");
        b_3.put("\u00CC", "\u00EC");
        b_3.put("\u00CD", "\u00ED");
        b_3.put("\u00CE", "\u00EE");
        b_3.put("\u00CF", "\u00EF");
        b_3.put("\u00D0", "\u00F0");
        b_3.put("\u00D1", "\u00F1");
        b_3.put("\u00D2", "\u00F2");
        b_3.put("\u00D3", "\u00F3");
        b_3.put("\u00D4", "\u00F4");
        b_3.put("\u00D5", "\u00F5");
        b_3.put("\u00D6", "\u00F6");
        b_3.put("\u00D8", "\u00F8");
        b_3.put("\u00D9", "\u00F9");
        b_3.put("\u00DA", "\u00FA");
        b_3.put("\u00DB", "\u00FB");
        b_3.put("\u00DC", "\u00FC");
        b_3.put("\u00DD", "\u00FD");
        b_3.put("\u00DE", "\u00FE");
        b_3.put("\u00DF", "\u0073\u0073");
        b_3.put("\u0100", "\u0101");
        b_3.put("\u0102", "\u0103");
        b_3.put("\u0104", "\u0105");
        b_3.put("\u0106", "\u0107");
        b_3.put("\u0108", "\u0109");
        b_3.put("\u010A", "\u010B");
        b_3.put("\u010C", "\u010D");
        b_3.put("\u010E", "\u010F");
        b_3.put("\u0110", "\u0111");
        b_3.put("\u0112", "\u0113");
        b_3.put("\u0114", "\u0115");
        b_3.put("\u0116", "\u0117");
        b_3.put("\u0118", "\u0119");
        b_3.put("\u011A", "\u011B");
        b_3.put("\u011C", "\u011D");
        b_3.put("\u011E", "\u011F");
        b_3.put("\u0120", "\u0121");
        b_3.put("\u0122", "\u0123");
        b_3.put("\u0124", "\u0125");
        b_3.put("\u0126", "\u0127");
        b_3.put("\u0128", "\u0129");
        b_3.put("\u012A", "\u012B");
        b_3.put("\u012C", "\u012D");
        b_3.put("\u012E", "\u012F");
        b_3.put("\u0130", "\u0069\u0307");
        b_3.put("\u0132", "\u0133");
        b_3.put("\u0134", "\u0135");
        b_3.put("\u0136", "\u0137");
        b_3.put("\u0139", "\u013A");
        b_3.put("\u013B", "\u013C");
        b_3.put("\u013D", "\u013E");
        b_3.put("\u013F", "\u0140");
        b_3.put("\u0141", "\u0142");
        b_3.put("\u0143", "\u0144");
        b_3.put("\u0145", "\u0146");
        b_3.put("\u0147", "\u0148");
        b_3.put("\u0149", "\u02BC\u006E");
        b_3.put("\u014A", "\u014B");
        b_3.put("\u014C", "\u014D");
        b_3.put("\u014E", "\u014F");
        b_3.put("\u0150", "\u0151");
        b_3.put("\u0152", "\u0153");
        b_3.put("\u0154", "\u0155");
        b_3.put("\u0156", "\u0157");
        b_3.put("\u0158", "\u0159");
        b_3.put("\u015A", "\u015B");
        b_3.put("\u015C", "\u015D");
        b_3.put("\u015E", "\u015F");
        b_3.put("\u0160", "\u0161");
        b_3.put("\u0162", "\u0163");
        b_3.put("\u0164", "\u0165");
        b_3.put("\u0166", "\u0167");
        b_3.put("\u0168", "\u0169");
        b_3.put("\u016A", "\u016B");
        b_3.put("\u016C", "\u016D");
        b_3.put("\u016E", "\u016F");
        b_3.put("\u0170", "\u0171");
        b_3.put("\u0172", "\u0173");
        b_3.put("\u0174", "\u0175");
        b_3.put("\u0176", "\u0177");
        b_3.put("\u0178", "\u00FF");
        b_3.put("\u0179", "\u017A");
        b_3.put("\u017B", "\u017C");
        b_3.put("\u017D", "\u017E");
        b_3.put("\u017F", "\u0073");
        b_3.put("\u0181", "\u0253");
        b_3.put("\u0182", "\u0183");
        b_3.put("\u0184", "\u0185");
        b_3.put("\u0186", "\u0254");
        b_3.put("\u0187", "\u0188");
        b_3.put("\u0189", "\u0256");
        b_3.put("\u018A", "\u0257");
        b_3.put("\u018B", "\u018C");
        b_3.put("\u018E", "\u01DD");
        b_3.put("\u018F", "\u0259");
        b_3.put("\u0190", "\u025B");
        b_3.put("\u0191", "\u0192");
        b_3.put("\u0193", "\u0260");
        b_3.put("\u0194", "\u0263");
        b_3.put("\u0196", "\u0269");
        b_3.put("\u0197", "\u0268");
        b_3.put("\u0198", "\u0199");
        b_3.put("\u019C", "\u026F");
        b_3.put("\u019D", "\u0272");
        b_3.put("\u019F", "\u0275");
        b_3.put("\u01A0", "\u01A1");
        b_3.put("\u01A2", "\u01A3");
        b_3.put("\u01A4", "\u01A5");
        b_3.put("\u01A6", "\u0280");
        b_3.put("\u01A7", "\u01A8");
        b_3.put("\u01A9", "\u0283");
        b_3.put("\u01AC", "\u01AD");
        b_3.put("\u01AE", "\u0288");
        b_3.put("\u01AF", "\u01B0");
        b_3.put("\u01B1", "\u028A");
        b_3.put("\u01B2", "\u028B");
        b_3.put("\u01B3", "\u01B4");
        b_3.put("\u01B5", "\u01B6");
        b_3.put("\u01B7", "\u0292");
        b_3.put("\u01B8", "\u01B9");
        b_3.put("\u01BC", "\u01BD");
        b_3.put("\u01C4", "\u01C6");
        b_3.put("\u01C5", "\u01C6");
        b_3.put("\u01C7", "\u01C9");
        b_3.put("\u01C8", "\u01C9");
        b_3.put("\u01CA", "\u01CC");
        b_3.put("\u01CB", "\u01CC");
        b_3.put("\u01CD", "\u01CE");
        b_3.put("\u01CF", "\u01D0");
        b_3.put("\u01D1", "\u01D2");
        b_3.put("\u01D3", "\u01D4");
        b_3.put("\u01D5", "\u01D6");
        b_3.put("\u01D7", "\u01D8");
        b_3.put("\u01D9", "\u01DA");
        b_3.put("\u01DB", "\u01DC");
        b_3.put("\u01DE", "\u01DF");
        b_3.put("\u01E0", "\u01E1");
        b_3.put("\u01E2", "\u01E3");
        b_3.put("\u01E4", "\u01E5");
        b_3.put("\u01E6", "\u01E7");
        b_3.put("\u01E8", "\u01E9");
        b_3.put("\u01EA", "\u01EB");
        b_3.put("\u01EC", "\u01ED");
        b_3.put("\u01EE", "\u01EF");
        b_3.put("\u01F0", "\u006A\u030C");
        b_3.put("\u01F1", "\u01F3");
        b_3.put("\u01F2", "\u01F3");
        b_3.put("\u01F4", "\u01F5");
        b_3.put("\u01F6", "\u0195");
        b_3.put("\u01F7", "\u01BF");
        b_3.put("\u01F8", "\u01F9");
        b_3.put("\u01FA", "\u01FB");
        b_3.put("\u01FC", "\u01FD");
        b_3.put("\u01FE", "\u01FF");
        b_3.put("\u0200", "\u0201");
        b_3.put("\u0202", "\u0203");
        b_3.put("\u0204", "\u0205");
        b_3.put("\u0206", "\u0207");
        b_3.put("\u0208", "\u0209");
        b_3.put("\u020A", "\u020B");
        b_3.put("\u020C", "\u020D");
        b_3.put("\u020E", "\u020F");
        b_3.put("\u0210", "\u0211");
        b_3.put("\u0212", "\u0213");
        b_3.put("\u0214", "\u0215");
        b_3.put("\u0216", "\u0217");
        b_3.put("\u0218", "\u0219");
        b_3.put("\u021A", "\u021B");
        b_3.put("\u021C", "\u021D");
        b_3.put("\u021E", "\u021F");
        b_3.put("\u0220", "\u019E");
        b_3.put("\u0222", "\u0223");
        b_3.put("\u0224", "\u0225");
        b_3.put("\u0226", "\u0227");
        b_3.put("\u0228", "\u0229");
        b_3.put("\u022A", "\u022B");
        b_3.put("\u022C", "\u022D");
        b_3.put("\u022E", "\u022F");
        b_3.put("\u0230", "\u0231");
        b_3.put("\u0232", "\u0233");
        b_3.put("\u0345", "\u03B9");
        b_3.put("\u0386", "\u03AC");
        b_3.put("\u0388", "\u03AD");
        b_3.put("\u0389", "\u03AE");
        b_3.put("\u038A", "\u03AF");
        b_3.put("\u038C", "\u03CC");
        b_3.put("\u038E", "\u03CD");
        b_3.put("\u038F", "\u03CE");
        b_3.put("\u0390", "\u03B9\u0308\u0301");
        b_3.put("\u0391", "\u03B1");
        b_3.put("\u0392", "\u03B2");
        b_3.put("\u0393", "\u03B3");
        b_3.put("\u0394", "\u03B4");
        b_3.put("\u0395", "\u03B5");
        b_3.put("\u0396", "\u03B6");
        b_3.put("\u0397", "\u03B7");
        b_3.put("\u0398", "\u03B8");
        b_3.put("\u0399", "\u03B9");
        b_3.put("\u039A", "\u03BA");
        b_3.put("\u039B", "\u03BB");
        b_3.put("\u039C", "\u03BC");
        b_3.put("\u039D", "\u03BD");
        b_3.put("\u039E", "\u03BE");
        b_3.put("\u039F", "\u03BF");
        b_3.put("\u03A0", "\u03C0");
        b_3.put("\u03A1", "\u03C1");
        b_3.put("\u03A3", "\u03C3");
        b_3.put("\u03A4", "\u03C4");
        b_3.put("\u03A5", "\u03C5");
        b_3.put("\u03A6", "\u03C6");
        b_3.put("\u03A7", "\u03C7");
        b_3.put("\u03A8", "\u03C8");
        b_3.put("\u03A9", "\u03C9");
        b_3.put("\u03AA", "\u03CA");
        b_3.put("\u03AB", "\u03CB");
        b_3.put("\u03B0", "\u03C5\u0308\u0301");
        b_3.put("\u03C2", "\u03C3");
        b_3.put("\u03D0", "\u03B2");
        b_3.put("\u03D1", "\u03B8");
        b_3.put("\u03D5", "\u03C6");
        b_3.put("\u03D6", "\u03C0");
        b_3.put("\u03D8", "\u03D9");
        b_3.put("\u03DA", "\u03DB");
        b_3.put("\u03DC", "\u03DD");
        b_3.put("\u03DE", "\u03DF");
        b_3.put("\u03E0", "\u03E1");
        b_3.put("\u03E2", "\u03E3");
        b_3.put("\u03E4", "\u03E5");
        b_3.put("\u03E6", "\u03E7");
        b_3.put("\u03E8", "\u03E9");
        b_3.put("\u03EA", "\u03EB");
        b_3.put("\u03EC", "\u03ED");
        b_3.put("\u03EE", "\u03EF");
        b_3.put("\u03F0", "\u03BA");
        b_3.put("\u03F1", "\u03C1");
        b_3.put("\u03F2", "\u03C3");
        b_3.put("\u03F4", "\u03B8");
        b_3.put("\u03F5", "\u03B5");
        b_3.put("\u0400", "\u0450");
        b_3.put("\u0401", "\u0451");
        b_3.put("\u0402", "\u0452");
        b_3.put("\u0403", "\u0453");
        b_3.put("\u0404", "\u0454");
        b_3.put("\u0405", "\u0455");
        b_3.put("\u0406", "\u0456");
        b_3.put("\u0407", "\u0457");
        b_3.put("\u0408", "\u0458");
        b_3.put("\u0409", "\u0459");
        b_3.put("\u040A", "\u045A");
        b_3.put("\u040B", "\u045B");
        b_3.put("\u040C", "\u045C");
        b_3.put("\u040D", "\u045D");
        b_3.put("\u040E", "\u045E");
        b_3.put("\u040F", "\u045F");
        b_3.put("\u0410", "\u0430");
        b_3.put("\u0411", "\u0431");
        b_3.put("\u0412", "\u0432");
        b_3.put("\u0413", "\u0433");
        b_3.put("\u0414", "\u0434");
        b_3.put("\u0415", "\u0435");
        b_3.put("\u0416", "\u0436");
        b_3.put("\u0417", "\u0437");
        b_3.put("\u0418", "\u0438");
        b_3.put("\u0419", "\u0439");
        b_3.put("\u041A", "\u043A");
        b_3.put("\u041B", "\u043B");
        b_3.put("\u041C", "\u043C");
        b_3.put("\u041D", "\u043D");
        b_3.put("\u041E", "\u043E");
        b_3.put("\u041F", "\u043F");
        b_3.put("\u0420", "\u0440");
        b_3.put("\u0421", "\u0441");
        b_3.put("\u0422", "\u0442");
        b_3.put("\u0423", "\u0443");
        b_3.put("\u0424", "\u0444");
        b_3.put("\u0425", "\u0445");
        b_3.put("\u0426", "\u0446");
        b_3.put("\u0427", "\u0447");
        b_3.put("\u0428", "\u0448");
        b_3.put("\u0429", "\u0449");
        b_3.put("\u042A", "\u044A");
        b_3.put("\u042B", "\u044B");
        b_3.put("\u042C", "\u044C");
        b_3.put("\u042D", "\u044D");
        b_3.put("\u042E", "\u044E");
        b_3.put("\u042F", "\u044F");
        b_3.put("\u0460", "\u0461");
        b_3.put("\u0462", "\u0463");
        b_3.put("\u0464", "\u0465");
        b_3.put("\u0466", "\u0467");
        b_3.put("\u0468", "\u0469");
        b_3.put("\u046A", "\u046B");
        b_3.put("\u046C", "\u046D");
        b_3.put("\u046E", "\u046F");
        b_3.put("\u0470", "\u0471");
        b_3.put("\u0472", "\u0473");
        b_3.put("\u0474", "\u0475");
        b_3.put("\u0476", "\u0477");
        b_3.put("\u0478", "\u0479");
        b_3.put("\u047A", "\u047B");
        b_3.put("\u047C", "\u047D");
        b_3.put("\u047E", "\u047F");
        b_3.put("\u0480", "\u0481");
        b_3.put("\u048A", "\u048B");
        b_3.put("\u048C", "\u048D");
        b_3.put("\u048E", "\u048F");
        b_3.put("\u0490", "\u0491");
        b_3.put("\u0492", "\u0493");
        b_3.put("\u0494", "\u0495");
        b_3.put("\u0496", "\u0497");
        b_3.put("\u0498", "\u0499");
        b_3.put("\u049A", "\u049B");
        b_3.put("\u049C", "\u049D");
        b_3.put("\u049E", "\u049F");
        b_3.put("\u04A0", "\u04A1");
        b_3.put("\u04A2", "\u04A3");
        b_3.put("\u04A4", "\u04A5");
        b_3.put("\u04A6", "\u04A7");
        b_3.put("\u04A8", "\u04A9");
        b_3.put("\u04AA", "\u04AB");
        b_3.put("\u04AC", "\u04AD");
        b_3.put("\u04AE", "\u04AF");
        b_3.put("\u04B0", "\u04B1");
        b_3.put("\u04B2", "\u04B3");
        b_3.put("\u04B4", "\u04B5");
        b_3.put("\u04B6", "\u04B7");
        b_3.put("\u04B8", "\u04B9");
        b_3.put("\u04BA", "\u04BB");
        b_3.put("\u04BC", "\u04BD");
        b_3.put("\u04BE", "\u04BF");
        b_3.put("\u04C1", "\u04C2");
        b_3.put("\u04C3", "\u04C4");
        b_3.put("\u04C5", "\u04C6");
        b_3.put("\u04C7", "\u04C8");
        b_3.put("\u04C9", "\u04CA");
        b_3.put("\u04CB", "\u04CC");
        b_3.put("\u04CD", "\u04CE");
        b_3.put("\u04D0", "\u04D1");
        b_3.put("\u04D2", "\u04D3");
        b_3.put("\u04D4", "\u04D5");
        b_3.put("\u04D6", "\u04D7");
        b_3.put("\u04D8", "\u04D9");
        b_3.put("\u04DA", "\u04DB");
        b_3.put("\u04DC", "\u04DD");
        b_3.put("\u04DE", "\u04DF");
        b_3.put("\u04E0", "\u04E1");
        b_3.put("\u04E2", "\u04E3");
        b_3.put("\u04E4", "\u04E5");
        b_3.put("\u04E6", "\u04E7");
        b_3.put("\u04E8", "\u04E9");
        b_3.put("\u04EA", "\u04EB");
        b_3.put("\u04EC", "\u04ED");
        b_3.put("\u04EE", "\u04EF");
        b_3.put("\u04F0", "\u04F1");
        b_3.put("\u04F2", "\u04F3");
        b_3.put("\u04F4", "\u04F5");
        b_3.put("\u04F8", "\u04F9");
        b_3.put("\u0500", "\u0501");
        b_3.put("\u0502", "\u0503");
        b_3.put("\u0504", "\u0505");
        b_3.put("\u0506", "\u0507");
        b_3.put("\u0508", "\u0509");
        b_3.put("\u050A", "\u050B");
        b_3.put("\u050C", "\u050D");
        b_3.put("\u050E", "\u050F");
        b_3.put("\u0531", "\u0561");
        b_3.put("\u0532", "\u0562");
        b_3.put("\u0533", "\u0563");
        b_3.put("\u0534", "\u0564");
        b_3.put("\u0535", "\u0565");
        b_3.put("\u0536", "\u0566");
        b_3.put("\u0537", "\u0567");
        b_3.put("\u0538", "\u0568");
        b_3.put("\u0539", "\u0569");
        b_3.put("\u053A", "\u056A");
        b_3.put("\u053B", "\u056B");
        b_3.put("\u053C", "\u056C");
        b_3.put("\u053D", "\u056D");
        b_3.put("\u053E", "\u056E");
        b_3.put("\u053F", "\u056F");
        b_3.put("\u0540", "\u0570");
        b_3.put("\u0541", "\u0571");
        b_3.put("\u0542", "\u0572");
        b_3.put("\u0543", "\u0573");
        b_3.put("\u0544", "\u0574");
        b_3.put("\u0545", "\u0575");
        b_3.put("\u0546", "\u0576");
        b_3.put("\u0547", "\u0577");
        b_3.put("\u0548", "\u0578");
        b_3.put("\u0549", "\u0579");
        b_3.put("\u054A", "\u057A");
        b_3.put("\u054B", "\u057B");
        b_3.put("\u054C", "\u057C");
        b_3.put("\u054D", "\u057D");
        b_3.put("\u054E", "\u057E");
        b_3.put("\u054F", "\u057F");
        b_3.put("\u0550", "\u0580");
        b_3.put("\u0551", "\u0581");
        b_3.put("\u0552", "\u0582");
        b_3.put("\u0553", "\u0583");
        b_3.put("\u0554", "\u0584");
        b_3.put("\u0555", "\u0585");
        b_3.put("\u0556", "\u0586");
        b_3.put("\u0587", "\u0565\u0582");
        b_3.put("\u1E00", "\u1E01");
        b_3.put("\u1E02", "\u1E03");
        b_3.put("\u1E04", "\u1E05");
        b_3.put("\u1E06", "\u1E07");
        b_3.put("\u1E08", "\u1E09");
        b_3.put("\u1E0A", "\u1E0B");
        b_3.put("\u1E0C", "\u1E0D");
        b_3.put("\u1E0E", "\u1E0F");
        b_3.put("\u1E10", "\u1E11");
        b_3.put("\u1E12", "\u1E13");
        b_3.put("\u1E14", "\u1E15");
        b_3.put("\u1E16", "\u1E17");
        b_3.put("\u1E18", "\u1E19");
        b_3.put("\u1E1A", "\u1E1B");
        b_3.put("\u1E1C", "\u1E1D");
        b_3.put("\u1E1E", "\u1E1F");
        b_3.put("\u1E20", "\u1E21");
        b_3.put("\u1E22", "\u1E23");
        b_3.put("\u1E24", "\u1E25");
        b_3.put("\u1E26", "\u1E27");
        b_3.put("\u1E28", "\u1E29");
        b_3.put("\u1E2A", "\u1E2B");
        b_3.put("\u1E2C", "\u1E2D");
        b_3.put("\u1E2E", "\u1E2F");
        b_3.put("\u1E30", "\u1E31");
        b_3.put("\u1E32", "\u1E33");
        b_3.put("\u1E34", "\u1E35");
        b_3.put("\u1E36", "\u1E37");
        b_3.put("\u1E38", "\u1E39");
        b_3.put("\u1E3A", "\u1E3B");
        b_3.put("\u1E3C", "\u1E3D");
        b_3.put("\u1E3E", "\u1E3F");
        b_3.put("\u1E40", "\u1E41");
        b_3.put("\u1E42", "\u1E43");
        b_3.put("\u1E44", "\u1E45");
        b_3.put("\u1E46", "\u1E47");
        b_3.put("\u1E48", "\u1E49");
        b_3.put("\u1E4A", "\u1E4B");
        b_3.put("\u1E4C", "\u1E4D");
        b_3.put("\u1E4E", "\u1E4F");
        b_3.put("\u1E50", "\u1E51");
        b_3.put("\u1E52", "\u1E53");
        b_3.put("\u1E54", "\u1E55");
        b_3.put("\u1E56", "\u1E57");
        b_3.put("\u1E58", "\u1E59");
        b_3.put("\u1E5A", "\u1E5B");
        b_3.put("\u1E5C", "\u1E5D");
        b_3.put("\u1E5E", "\u1E5F");
        b_3.put("\u1E60", "\u1E61");
        b_3.put("\u1E62", "\u1E63");
        b_3.put("\u1E64", "\u1E65");
        b_3.put("\u1E66", "\u1E67");
        b_3.put("\u1E68", "\u1E69");
        b_3.put("\u1E6A", "\u1E6B");
        b_3.put("\u1E6C", "\u1E6D");
        b_3.put("\u1E6E", "\u1E6F");
        b_3.put("\u1E70", "\u1E71");
        b_3.put("\u1E72", "\u1E73");
        b_3.put("\u1E74", "\u1E75");
        b_3.put("\u1E76", "\u1E77");
        b_3.put("\u1E78", "\u1E79");
        b_3.put("\u1E7A", "\u1E7B");
        b_3.put("\u1E7C", "\u1E7D");
        b_3.put("\u1E7E", "\u1E7F");
        b_3.put("\u1E80", "\u1E81");
        b_3.put("\u1E82", "\u1E83");
        b_3.put("\u1E84", "\u1E85");
        b_3.put("\u1E86", "\u1E87");
        b_3.put("\u1E88", "\u1E89");
        b_3.put("\u1E8A", "\u1E8B");
        b_3.put("\u1E8C", "\u1E8D");
        b_3.put("\u1E8E", "\u1E8F");
        b_3.put("\u1E90", "\u1E91");
        b_3.put("\u1E92", "\u1E93");
        b_3.put("\u1E94", "\u1E95");
        b_3.put("\u1E96", "\u0068\u0331");
        b_3.put("\u1E97", "\u0074\u0308");
        b_3.put("\u1E98", "\u0077\u030A");
        b_3.put("\u1E99", "\u0079\u030A");
        b_3.put("\u1E9A", "\u0061\u02BE");
        b_3.put("\u1E9B", "\u1E61");
        b_3.put("\u1EA0", "\u1EA1");
        b_3.put("\u1EA2", "\u1EA3");
        b_3.put("\u1EA4", "\u1EA5");
        b_3.put("\u1EA6", "\u1EA7");
        b_3.put("\u1EA8", "\u1EA9");
        b_3.put("\u1EAA", "\u1EAB");
        b_3.put("\u1EAC", "\u1EAD");
        b_3.put("\u1EAE", "\u1EAF");
        b_3.put("\u1EB0", "\u1EB1");
        b_3.put("\u1EB2", "\u1EB3");
        b_3.put("\u1EB4", "\u1EB5");
        b_3.put("\u1EB6", "\u1EB7");
        b_3.put("\u1EB8", "\u1EB9");
        b_3.put("\u1EBA", "\u1EBB");
        b_3.put("\u1EBC", "\u1EBD");
        b_3.put("\u1EBE", "\u1EBF");
        b_3.put("\u1EC0", "\u1EC1");
        b_3.put("\u1EC2", "\u1EC3");
        b_3.put("\u1EC4", "\u1EC5");
        b_3.put("\u1EC6", "\u1EC7");
        b_3.put("\u1EC8", "\u1EC9");
        b_3.put("\u1ECA", "\u1ECB");
        b_3.put("\u1ECC", "\u1ECD");
        b_3.put("\u1ECE", "\u1ECF");
        b_3.put("\u1ED0", "\u1ED1");
        b_3.put("\u1ED2", "\u1ED3");
        b_3.put("\u1ED4", "\u1ED5");
        b_3.put("\u1ED6", "\u1ED7");
        b_3.put("\u1ED8", "\u1ED9");
        b_3.put("\u1EDA", "\u1EDB");
        b_3.put("\u1EDC", "\u1EDD");
        b_3.put("\u1EDE", "\u1EDF");
        b_3.put("\u1EE0", "\u1EE1");
        b_3.put("\u1EE2", "\u1EE3");
        b_3.put("\u1EE4", "\u1EE5");
        b_3.put("\u1EE6", "\u1EE7");
        b_3.put("\u1EE8", "\u1EE9");
        b_3.put("\u1EEA", "\u1EEB");
        b_3.put("\u1EEC", "\u1EED");
        b_3.put("\u1EEE", "\u1EEF");
        b_3.put("\u1EF0", "\u1EF1");
        b_3.put("\u1EF2", "\u1EF3");
        b_3.put("\u1EF4", "\u1EF5");
        b_3.put("\u1EF6", "\u1EF7");
        b_3.put("\u1EF8", "\u1EF9");
        b_3.put("\u1F08", "\u1F00");
        b_3.put("\u1F09", "\u1F01");
        b_3.put("\u1F0A", "\u1F02");
        b_3.put("\u1F0B", "\u1F03");
        b_3.put("\u1F0C", "\u1F04");
        b_3.put("\u1F0D", "\u1F05");
        b_3.put("\u1F0E", "\u1F06");
        b_3.put("\u1F0F", "\u1F07");
        b_3.put("\u1F18", "\u1F10");
        b_3.put("\u1F19", "\u1F11");
        b_3.put("\u1F1A", "\u1F12");
        b_3.put("\u1F1B", "\u1F13");
        b_3.put("\u1F1C", "\u1F14");
        b_3.put("\u1F1D", "\u1F15");
        b_3.put("\u1F28", "\u1F20");
        b_3.put("\u1F29", "\u1F21");
        b_3.put("\u1F2A", "\u1F22");
        b_3.put("\u1F2B", "\u1F23");
        b_3.put("\u1F2C", "\u1F24");
        b_3.put("\u1F2D", "\u1F25");
        b_3.put("\u1F2E", "\u1F26");
        b_3.put("\u1F2F", "\u1F27");
        b_3.put("\u1F38", "\u1F30");
        b_3.put("\u1F39", "\u1F31");
        b_3.put("\u1F3A", "\u1F32");
        b_3.put("\u1F3B", "\u1F33");
        b_3.put("\u1F3C", "\u1F34");
        b_3.put("\u1F3D", "\u1F35");
        b_3.put("\u1F3E", "\u1F36");
        b_3.put("\u1F3F", "\u1F37");
        b_3.put("\u1F48", "\u1F40");
        b_3.put("\u1F49", "\u1F41");
        b_3.put("\u1F4A", "\u1F42");
        b_3.put("\u1F4B", "\u1F43");
        b_3.put("\u1F4C", "\u1F44");
        b_3.put("\u1F4D", "\u1F45");
        b_3.put("\u1F50", "\u03C5\u0313");
        b_3.put("\u1F52", "\u03C5\u0313\u0300");
        b_3.put("\u1F54", "\u03C5\u0313\u0301");
        b_3.put("\u1F56", "\u03C5\u0313\u0342");
        b_3.put("\u1F59", "\u1F51");
        b_3.put("\u1F5B", "\u1F53");
        b_3.put("\u1F5D", "\u1F55");
        b_3.put("\u1F5F", "\u1F57");
        b_3.put("\u1F68", "\u1F60");
        b_3.put("\u1F69", "\u1F61");
        b_3.put("\u1F6A", "\u1F62");
        b_3.put("\u1F6B", "\u1F63");
        b_3.put("\u1F6C", "\u1F64");
        b_3.put("\u1F6D", "\u1F65");
        b_3.put("\u1F6E", "\u1F66");
        b_3.put("\u1F6F", "\u1F67");
        b_3.put("\u1F80", "\u1F00\u03B9");
        b_3.put("\u1F81", "\u1F01\u03B9");
        b_3.put("\u1F82", "\u1F02\u03B9");
        b_3.put("\u1F83", "\u1F03\u03B9");
        b_3.put("\u1F84", "\u1F04\u03B9");
        b_3.put("\u1F85", "\u1F05\u03B9");
        b_3.put("\u1F86", "\u1F06\u03B9");
        b_3.put("\u1F87", "\u1F07\u03B9");
        b_3.put("\u1F88", "\u1F00\u03B9");
        b_3.put("\u1F89", "\u1F01\u03B9");
        b_3.put("\u1F8A", "\u1F02\u03B9");
        b_3.put("\u1F8B", "\u1F03\u03B9");
        b_3.put("\u1F8C", "\u1F04\u03B9");
        b_3.put("\u1F8D", "\u1F05\u03B9");
        b_3.put("\u1F8E", "\u1F06\u03B9");
        b_3.put("\u1F8F", "\u1F07\u03B9");
        b_3.put("\u1F90", "\u1F20\u03B9");
        b_3.put("\u1F91", "\u1F21\u03B9");
        b_3.put("\u1F92", "\u1F22\u03B9");
        b_3.put("\u1F93", "\u1F23\u03B9");
        b_3.put("\u1F94", "\u1F24\u03B9");
        b_3.put("\u1F95", "\u1F25\u03B9");
        b_3.put("\u1F96", "\u1F26\u03B9");
        b_3.put("\u1F97", "\u1F27\u03B9");
        b_3.put("\u1F98", "\u1F20\u03B9");
        b_3.put("\u1F99", "\u1F21\u03B9");
        b_3.put("\u1F9A", "\u1F22\u03B9");
        b_3.put("\u1F9B", "\u1F23\u03B9");
        b_3.put("\u1F9C", "\u1F24\u03B9");
        b_3.put("\u1F9D", "\u1F25\u03B9");
        b_3.put("\u1F9E", "\u1F26\u03B9");
        b_3.put("\u1F9F", "\u1F27\u03B9");
        b_3.put("\u1FA0", "\u1F60\u03B9");
        b_3.put("\u1FA1", "\u1F61\u03B9");
        b_3.put("\u1FA2", "\u1F62\u03B9");
        b_3.put("\u1FA3", "\u1F63\u03B9");
        b_3.put("\u1FA4", "\u1F64\u03B9");
        b_3.put("\u1FA5", "\u1F65\u03B9");
        b_3.put("\u1FA6", "\u1F66\u03B9");
        b_3.put("\u1FA7", "\u1F67\u03B9");
        b_3.put("\u1FA8", "\u1F60\u03B9");
        b_3.put("\u1FA9", "\u1F61\u03B9");
        b_3.put("\u1FAA", "\u1F62\u03B9");
        b_3.put("\u1FAB", "\u1F63\u03B9");
        b_3.put("\u1FAC", "\u1F64\u03B9");
        b_3.put("\u1FAD", "\u1F65\u03B9");
        b_3.put("\u1FAE", "\u1F66\u03B9");
        b_3.put("\u1FAF", "\u1F67\u03B9");
        b_3.put("\u1FB2", "\u1F70\u03B9");
        b_3.put("\u1FB3", "\u03B1\u03B9");
        b_3.put("\u1FB4", "\u03AC\u03B9");
        b_3.put("\u1FB6", "\u03B1\u0342");
        b_3.put("\u1FB7", "\u03B1\u0342\u03B9");
        b_3.put("\u1FB8", "\u1FB0");
        b_3.put("\u1FB9", "\u1FB1");
        b_3.put("\u1FBA", "\u1F70");
        b_3.put("\u1FBB", "\u1F71");
        b_3.put("\u1FBC", "\u03B1\u03B9");
        b_3.put("\u1FBE", "\u03B9");
        b_3.put("\u1FC2", "\u1F74\u03B9");
        b_3.put("\u1FC3", "\u03B7\u03B9");
        b_3.put("\u1FC4", "\u03AE\u03B9");
        b_3.put("\u1FC6", "\u03B7\u0342");
        b_3.put("\u1FC7", "\u03B7\u0342\u03B9");
        b_3.put("\u1FC8", "\u1F72");
        b_3.put("\u1FC9", "\u1F73");
        b_3.put("\u1FCA", "\u1F74");
        b_3.put("\u1FCB", "\u1F75");
        b_3.put("\u1FCC", "\u03B7\u03B9");
        b_3.put("\u1FD2", "\u03B9\u0308\u0300");
        b_3.put("\u1FD3", "\u03B9\u0308\u0301");
        b_3.put("\u1FD6", "\u03B9\u0342");
        b_3.put("\u1FD7", "\u03B9\u0308\u0342");
        b_3.put("\u1FD8", "\u1FD0");
        b_3.put("\u1FD9", "\u1FD1");
        b_3.put("\u1FDA", "\u1F76");
        b_3.put("\u1FDB", "\u1F77");
        b_3.put("\u1FE2", "\u03C5\u0308\u0300");
        b_3.put("\u1FE3", "\u03C5\u0308\u0301");
        b_3.put("\u1FE4", "\u03C1\u0313");
        b_3.put("\u1FE6", "\u03C5\u0342");
        b_3.put("\u1FE7", "\u03C5\u0308\u0342");
        b_3.put("\u1FE8", "\u1FE0");
        b_3.put("\u1FE9", "\u1FE1");
        b_3.put("\u1FEA", "\u1F7A");
        b_3.put("\u1FEB", "\u1F7B");
        b_3.put("\u1FEC", "\u1FE5");
        b_3.put("\u1FF2", "\u1F7C\u03B9");
        b_3.put("\u1FF3", "\u03C9\u03B9");
        b_3.put("\u1FF4", "\u03CE\u03B9");
        b_3.put("\u1FF6", "\u03C9\u0342");
        b_3.put("\u1FF7", "\u03C9\u0342\u03B9");
        b_3.put("\u1FF8", "\u1F78");
        b_3.put("\u1FF9", "\u1F79");
        b_3.put("\u1FFA", "\u1F7C");
        b_3.put("\u1FFB", "\u1F7D");
        b_3.put("\u1FFC", "\u03C9\u03B9");
        b_3.put("\u2126", "\u03C9");
        b_3.put("\u212A", "\u006B");
        b_3.put("\u212B", "\u00E5");
        b_3.put("\u2160", "\u2170");
        b_3.put("\u2161", "\u2171");
        b_3.put("\u2162", "\u2172");
        b_3.put("\u2163", "\u2173");
        b_3.put("\u2164", "\u2174");
        b_3.put("\u2165", "\u2175");
        b_3.put("\u2166", "\u2176");
        b_3.put("\u2167", "\u2177");
        b_3.put("\u2168", "\u2178");
        b_3.put("\u2169", "\u2179");
        b_3.put("\u216A", "\u217A");
        b_3.put("\u216B", "\u217B");
        b_3.put("\u216C", "\u217C");
        b_3.put("\u216D", "\u217D");
        b_3.put("\u216E", "\u217E");
        b_3.put("\u216F", "\u217F");
        b_3.put("\u24B6", "\u24D0");
        b_3.put("\u24B7", "\u24D1");
        b_3.put("\u24B8", "\u24D2");
        b_3.put("\u24B9", "\u24D3");
        b_3.put("\u24BA", "\u24D4");
        b_3.put("\u24BB", "\u24D5");
        b_3.put("\u24BC", "\u24D6");
        b_3.put("\u24BD", "\u24D7");
        b_3.put("\u24BE", "\u24D8");
        b_3.put("\u24BF", "\u24D9");
        b_3.put("\u24C0", "\u24DA");
        b_3.put("\u24C1", "\u24DB");
        b_3.put("\u24C2", "\u24DC");
        b_3.put("\u24C3", "\u24DD");
        b_3.put("\u24C4", "\u24DE");
        b_3.put("\u24C5", "\u24DF");
        b_3.put("\u24C6", "\u24E0");
        b_3.put("\u24C7", "\u24E1");
        b_3.put("\u24C8", "\u24E2");
        b_3.put("\u24C9", "\u24E3");
        b_3.put("\u24CA", "\u24E4");
        b_3.put("\u24CB", "\u24E5");
        b_3.put("\u24CC", "\u24E6");
        b_3.put("\u24CD", "\u24E7");
        b_3.put("\u24CE", "\u24E8");
        b_3.put("\u24CF", "\u24E9");
        b_3.put("\uFB00", "\u0066\u0066");
        b_3.put("\uFB01", "\u0066\u0069");
        b_3.put("\uFB02", "\u0066\u006C");
        b_3.put("\uFB03", "\u0066\u0066\u0069");
        b_3.put("\uFB04", "\u0066\u0066\u006C");
        b_3.put("\uFB05", "\u0073\u0074");
        b_3.put("\uFB06", "\u0073\u0074");
        b_3.put("\uFB13", "\u0574\u0576");
        b_3.put("\uFB14", "\u0574\u0565");
        b_3.put("\uFB15", "\u0574\u056B");
        b_3.put("\uFB16", "\u057E\u0576");
        b_3.put("\uFB17", "\u0574\u056D");
        b_3.put("\uFF21", "\uFF41");
        b_3.put("\uFF22", "\uFF42");
        b_3.put("\uFF23", "\uFF43");
        b_3.put("\uFF24", "\uFF44");
        b_3.put("\uFF25", "\uFF45");
        b_3.put("\uFF26", "\uFF46");
        b_3.put("\uFF27", "\uFF47");
        b_3.put("\uFF28", "\uFF48");
        b_3.put("\uFF29", "\uFF49");
        b_3.put("\uFF2A", "\uFF4A");
        b_3.put("\uFF2B", "\uFF4B");
        b_3.put("\uFF2C", "\uFF4C");
        b_3.put("\uFF2D", "\uFF4D");
        b_3.put("\uFF2E", "\uFF4E");
        b_3.put("\uFF2F", "\uFF4F");
        b_3.put("\uFF30", "\uFF50");
        b_3.put("\uFF31", "\uFF51");
        b_3.put("\uFF32", "\uFF52");
        b_3.put("\uFF33", "\uFF53");
        b_3.put("\uFF34", "\uFF54");
        b_3.put("\uFF35", "\uFF55");
        b_3.put("\uFF36", "\uFF56");
        b_3.put("\uFF37", "\uFF57");
        b_3.put("\uFF38", "\uFF58");
        b_3.put("\uFF39", "\uFF59");
        b_3.put("\uFF3A", "\uFF5A");
        b_3.put(codePointToString(0x10400), codePointToString(0x10428));
        b_3.put(codePointToString(0x10401), codePointToString(0x10429));
        b_3.put(codePointToString(0x10402), codePointToString(0x1042A));
        b_3.put(codePointToString(0x10403), codePointToString(0x1042B));
        b_3.put(codePointToString(0x10404), codePointToString(0x1042C));
        b_3.put(codePointToString(0x10405), codePointToString(0x1042D));
        b_3.put(codePointToString(0x10406), codePointToString(0x1042E));
        b_3.put(codePointToString(0x10407), codePointToString(0x1042F));
        b_3.put(codePointToString(0x10408), codePointToString(0x10430));
        b_3.put(codePointToString(0x10409), codePointToString(0x10431));
        b_3.put(codePointToString(0x1040A), codePointToString(0x10432));
        b_3.put(codePointToString(0x1040B), codePointToString(0x10433));
        b_3.put(codePointToString(0x1040C), codePointToString(0x10434));
        b_3.put(codePointToString(0x1040D), codePointToString(0x10435));
        b_3.put(codePointToString(0x1040E), codePointToString(0x10436));
        b_3.put(codePointToString(0x1040F), codePointToString(0x10437));
        b_3.put(codePointToString(0x10410), codePointToString(0x10438));
        b_3.put(codePointToString(0x10411), codePointToString(0x10439));
        b_3.put(codePointToString(0x10412), codePointToString(0x1043A));
        b_3.put(codePointToString(0x10413), codePointToString(0x1043B));
        b_3.put(codePointToString(0x10414), codePointToString(0x1043C));
        b_3.put(codePointToString(0x10415), codePointToString(0x1043D));
        b_3.put(codePointToString(0x10416), codePointToString(0x1043E));
        b_3.put(codePointToString(0x10417), codePointToString(0x1043F));
        b_3.put(codePointToString(0x10418), codePointToString(0x10440));
        b_3.put(codePointToString(0x10419), codePointToString(0x10441));
        b_3.put(codePointToString(0x1041A), codePointToString(0x10442));
        b_3.put(codePointToString(0x1041B), codePointToString(0x10443));
        b_3.put(codePointToString(0x1041C), codePointToString(0x10444));
        b_3.put(codePointToString(0x1041D), codePointToString(0x10445));
        b_3.put(codePointToString(0x1041E), codePointToString(0x10446));
        b_3.put(codePointToString(0x1041F), codePointToString(0x10447));
        b_3.put(codePointToString(0x10420), codePointToString(0x10448));
        b_3.put(codePointToString(0x10421), codePointToString(0x10449));
        b_3.put(codePointToString(0x10422), codePointToString(0x1044A));
        b_3.put(codePointToString(0x10423), codePointToString(0x1044B));
        b_3.put(codePointToString(0x10424), codePointToString(0x1044C));
        b_3.put(codePointToString(0x10425), codePointToString(0x1044D));
        return Collections.unmodifiableMap(b_3);
    }

    private static final Set<String> build_C_1_1_AsciiSpaceCharacters() {
        Set<String> c_1_1 = new HashSet<String>();
        c_1_1.add("\u0020");
        return Collections.unmodifiableSet(c_1_1);
    }

    private static final Set<String> build_C_1_2_NonAsciiSpaceCharacters() {
        Set<String> c_1_2 = new HashSet<String>();
        c_1_2.add("\u00A0");
        c_1_2.add("\u1680");
        c_1_2.add("\u2000");
        c_1_2.add("\u2001");
        c_1_2.add("\u2002");
        c_1_2.add("\u2003");
        c_1_2.add("\u2004");
        c_1_2.add("\u2005");
        c_1_2.add("\u2006");
        c_1_2.add("\u2007");
        c_1_2.add("\u2008");
        c_1_2.add("\u2009");
        c_1_2.add("\u200A");
        c_1_2.add("\u200B");
        c_1_2.add("\u202F");
        c_1_2.add("\u205F");
        c_1_2.add("\u2000");
        return Collections.unmodifiableSet(c_1_2);
    }

    private static final Set<String> build_C_2_1_AsciiControlCharacters() {
        Set<String> c_2_1 = new HashSet<String>();
        addRange(c_2_1, 0x0000, 0x001F);
        c_2_1.add("\u007F");
        return Collections.unmodifiableSet(c_2_1);
    }

    private static final Set<String> build_C_2_2_NonAsciiControlCharacters() {
        Set<String> c_2_2 = new HashSet<String>();
        addRange(c_2_2, 0x0080, 0x009F);
        c_2_2.add("\u06DD");
        c_2_2.add("\u070F");
        c_2_2.add("\u180E");
        c_2_2.add("\u200C");
        c_2_2.add("\u200D");
        c_2_2.add("\u2028");
        c_2_2.add("\u2029");
        addRange(c_2_2, 0x2060, 0x2063);
        addRange(c_2_2, 0x206A, 0x206F);
        c_2_2.add("\uFeFF");
        addRange(c_2_2, 0xFFF9, 0xFFFC);
        addRange(c_2_2, 0x1D173, 0x1D17A);
        return Collections.unmodifiableSet(c_2_2);
    }

    private static final Set<String> build_C_3_PrivateUse() {
        Set<String> c_3 = new HashSet<String>();
        addRange(c_3, 0xE000, 0xF8FF);
        addRange(c_3, 0xF0000, 0xFFFFD);
        addRange(c_3, 0x100000, 0x10FFFD);
        return Collections.unmodifiableSet(c_3);
    }

    private static final Set<String> build_C_4_NonCharacterCodePoints() {
        Set<String> c_4 = new HashSet<String>();
        addRange(c_4, 0xFDD0, 0xFDEF);
        addRange(c_4, 0xFFFE, 0xFFFF);
        addRange(c_4, 0x1FFFE, 0x1FFFF);
        addRange(c_4, 0x2FFFE, 0x2FFFF);
        addRange(c_4, 0x3FFFE, 0x3FFFF);
        addRange(c_4, 0x4FFFE, 0x4FFFF);
        addRange(c_4, 0x5FFFE, 0x5FFFF);
        addRange(c_4, 0x6FFFE, 0x6FFFF);
        addRange(c_4, 0x7FFFE, 0x7FFFF);
        addRange(c_4, 0x8FFFE, 0x8FFFF);
        addRange(c_4, 0x9FFFE, 0x9FFFF);
        addRange(c_4, 0xAFFFE, 0xAFFFF);
        addRange(c_4, 0xBFFFE, 0xBFFFF);
        addRange(c_4, 0xCFFFE, 0xCFFFF);
        addRange(c_4, 0xDFFFE, 0xDFFFF);
        addRange(c_4, 0xEFFFE, 0xEFFFF);
        addRange(c_4, 0xFFFFE, 0xFFFFF);
        addRange(c_4, 0x10FFFE, 0x10FFFF);
        return Collections.unmodifiableSet(c_4);
    }

    private static final Set<String> build_C_5_SurrogateCodes() {
        Set<String> c_5 = new HashSet<String>();
        addRange(c_5, 0xD800, 0xDFFF);
        return Collections.unmodifiableSet(c_5);
    }

    private static final Set<String> build_C_6_InappropriateForPlainText() {
        Set<String> c_6 = new HashSet<String>();
        addRange(c_6, 0xFFF9, 0xFFFD);
        return Collections.unmodifiableSet(c_6);
    }

    private static final Set<String> build_C_7_InappropriateForCanonicalRepresentation() {
        Set<String> c_7 = new HashSet<String>();
        addRange(c_7, 0x2FF0, 0x2FFB);
        return Collections.unmodifiableSet(c_7);
    }

    private static final Set<String> build_C_8_ChangeDisplayPropertiesOrAreDeprecated() {
        Set<String> c_8 = new HashSet<String>();
        c_8.add("\u0340");
        c_8.add("\u0341");
        c_8.add("\u200E");
        c_8.add("\u200F");
        addRange(c_8, 0x202A, 0x202E);
        addRange(c_8, 0x206A, 0x206F);
        return Collections.unmodifiableSet(c_8);
    }

    private static final Set<String> build_C_9_TaggingCharacters() {
        Set<String> c_9 = new HashSet<String>();
        c_9.add(codePointToString(0xE0001));
        addRange(c_9, 0xE0020, 0xE007F);
        return Collections.unmodifiableSet(c_9);
    }

    private static final Set<String> build_D_1_CharactersWithBiDiPropertiesRorAl() {
        Set<String> RandAlCat = new HashSet<String>();
        RandAlCat.add("\u05BE");
        RandAlCat.add("\u05C0");
        RandAlCat.add("\u05C3");
        addRange(RandAlCat, 0x05D0, 0x05EA);
        addRange(RandAlCat, 0x05F0, 0x05F4);
        RandAlCat.add("\u061B");
        RandAlCat.add("\u061F");
        addRange(RandAlCat, 0x0621, 0x063A);
        addRange(RandAlCat, 0x0640, 0x064A);
        addRange(RandAlCat, 0x066D, 0x066F);
        addRange(RandAlCat, 0x0671, 0x06D5);
        RandAlCat.add("\u06DD");
        addRange(RandAlCat, 0x06E5, 0x06E6);
        addRange(RandAlCat, 0x06FA, 0x06FE);
        addRange(RandAlCat, 0x0700, 0x070D);
        RandAlCat.add("\u0710");
        addRange(RandAlCat, 0x0712, 0x072C);
        addRange(RandAlCat, 0x0780, 0x07A5);
        RandAlCat.add("\u07B1");
        RandAlCat.add("\u200F");
        RandAlCat.add("\uFB1D");
        addRange(RandAlCat, 0xFB1F, 0xFB28);
        addRange(RandAlCat, 0xFB2A, 0xFB36);
        addRange(RandAlCat, 0xFB38, 0xFB3C);
        RandAlCat.add("\uFB3E");
        addRange(RandAlCat, 0xFB40, 0xFB41);
        addRange(RandAlCat, 0xFB43, 0xFB44);
        addRange(RandAlCat, 0xFB46, 0xFBB1);
        addRange(RandAlCat, 0xFBD3, 0xFD3D);
        addRange(RandAlCat, 0xFD50, 0xFD8F);
        addRange(RandAlCat, 0xFD92, 0xFDC7);
        addRange(RandAlCat, 0xFDF0, 0xFDFC);
        addRange(RandAlCat, 0xFE70, 0xFE74);
        addRange(RandAlCat, 0xFE76, 0xFEFC);
        return Collections.unmodifiableSet(RandAlCat);
    }

    private static final Set<String> build_D_2_CharactersWithBiDiPropertyL() {
        Set<String> LCat = new HashSet<String>();
        addRange(LCat, 0x0041, 0x005A);
        addRange(LCat, 0x0061, 0x007A);
        LCat.add("\u00AA");
        LCat.add("\u00B5");
        LCat.add("\u00BA");
        addRange(LCat, 0x00C0, 0x00D6);
        addRange(LCat, 0x00D8, 0x00F6);
        addRange(LCat, 0x00F8, 0x0220);
        addRange(LCat, 0x0222, 0x0233);
        addRange(LCat, 0x0250, 0x02AD);
        addRange(LCat, 0x02B0, 0x02B8);
        addRange(LCat, 0x02BB, 0x02C1);
        addRange(LCat, 0x02D0, 0x02D1);
        addRange(LCat, 0x02E0, 0x02E4);
        LCat.add("\u02EE");
        LCat.add("\u037A");
        LCat.add("\u0386");
        addRange(LCat, 0x0388, 0x038A);
        LCat.add("\u038C");
        addRange(LCat, 0x038E, 0x03A1);
        addRange(LCat, 0x03A3, 0x03CE);
        addRange(LCat, 0x03D0, 0x03F5);
        addRange(LCat, 0x0400, 0x0482);
        addRange(LCat, 0x048A, 0x04CE);
        addRange(LCat, 0x04D0, 0x04F5);
        addRange(LCat, 0x04F8, 0x04F9);
        addRange(LCat, 0x0500, 0x050F);
        addRange(LCat, 0x0531, 0x0556);
        addRange(LCat, 0x0559, 0x055F);
        addRange(LCat, 0x0561, 0x0587);
        LCat.add("\u0589");
        LCat.add("\u0903");
        addRange(LCat, 0x0905, 0x0939);
        addRange(LCat, 0x093D, 0x0940);
        addRange(LCat, 0x0949, 0x094C);
        LCat.add("\u0950");
        addRange(LCat, 0x0958, 0x0961);
        addRange(LCat, 0x0964, 0x0970);
        addRange(LCat, 0x0982, 0x0983);
        addRange(LCat, 0x0985, 0x098C);
        addRange(LCat, 0x098F, 0x0990);
        addRange(LCat, 0x0993, 0x09A8);
        addRange(LCat, 0x09AA, 0x09B0);
        LCat.add("\u09B2");
        addRange(LCat, 0x09B6, 0x09B9);
        addRange(LCat, 0x09BE, 0x09C0);
        addRange(LCat, 0x09C7, 0x09C8);
        addRange(LCat, 0x09CB, 0x09CC);
        LCat.add("\u09D7");
        addRange(LCat, 0x09DC, 0x09DD);
        addRange(LCat, 0x09DF, 0x09E1);
        addRange(LCat, 0x09E6, 0x09F1);
        addRange(LCat, 0x09F4, 0x09FA);
        addRange(LCat, 0x0A05, 0x0A0A);
        addRange(LCat, 0x0A0F, 0x0A10);
        addRange(LCat, 0x0A13, 0x0A28);
        addRange(LCat, 0x0A2A, 0x0A30);
        addRange(LCat, 0x0A32, 0x0A33);
        addRange(LCat, 0x0A35, 0x0A36);
        addRange(LCat, 0x0A38, 0x0A39);
        addRange(LCat, 0x0A3E, 0x0A40);
        addRange(LCat, 0x0A59, 0x0A5C);
        LCat.add("\u0A5E");
        addRange(LCat, 0x0A66, 0x0A6F);
        addRange(LCat, 0x0A72, 0x0A74);
        LCat.add("\u0A83");
        addRange(LCat, 0x0A85, 0x0A8B);
        LCat.add("\u0A8D");
        addRange(LCat, 0x0A8F, 0x0A91);
        addRange(LCat, 0x0A93, 0x0AA8);
        addRange(LCat, 0x0AAA, 0x0AB0);
        addRange(LCat, 0x0AB2, 0x0AB3);
        addRange(LCat, 0x0AB5, 0x0AB9);
        addRange(LCat, 0x0ABD, 0x0AC0);
        LCat.add("\u0AC9");
        addRange(LCat, 0x0ACB, 0x0ACC);
        LCat.add("\u0AD0");
        LCat.add("\u0AE0");
        addRange(LCat, 0x0AE6, 0x0AEF);
        addRange(LCat, 0x0B02, 0x0B03);
        addRange(LCat, 0x0B05, 0x0B0C);
        addRange(LCat, 0x0B0F, 0x0B10);
        addRange(LCat, 0x0B13, 0x0B28);
        addRange(LCat, 0x0B2A, 0x0B30);
        addRange(LCat, 0x0B32, 0x0B33);
        addRange(LCat, 0x0B36, 0x0B39);
        addRange(LCat, 0x0B3D, 0x0B3E);
        LCat.add("\u0B40");
        addRange(LCat, 0x0B47, 0x0B48);
        addRange(LCat, 0x0B4B, 0x0B4C);
        LCat.add("\u0B57");
        addRange(LCat, 0x0B5C, 0x0B5D);
        addRange(LCat, 0x0B5F, 0x0B61);
        addRange(LCat, 0x0B66, 0x0B70);
        LCat.add("\u0B83");
        addRange(LCat, 0x0B85, 0x0B8A);
        addRange(LCat, 0x0B8E, 0x0B90);
        addRange(LCat, 0x0B92, 0x0B95);
        addRange(LCat, 0x0B99, 0x0B9A);
        LCat.add("\u0B9C");
        addRange(LCat, 0x0B9E, 0x0B9F);
        addRange(LCat, 0x0BA3, 0x0BA4);
        addRange(LCat, 0x0BA8, 0x0BAA);
        addRange(LCat, 0x0BAE, 0x0BB5);
        addRange(LCat, 0x0BB7, 0x0BB9);
        addRange(LCat, 0x0BBE, 0x0BBF);
        addRange(LCat, 0x0BC1, 0x0BC2);
        addRange(LCat, 0x0BC6, 0x0BC8);
        addRange(LCat, 0x0BCA, 0x0BCC);
        LCat.add("\u0BD7");
        addRange(LCat, 0x0BE7, 0x0BF2);
        addRange(LCat, 0x0C01, 0x0C03);
        addRange(LCat, 0x0C05, 0x0C0C);
        addRange(LCat, 0x0C0E, 0x0C10);
        addRange(LCat, 0x0C12, 0x0C28);
        addRange(LCat, 0x0C2A, 0x0C33);
        addRange(LCat, 0x0C35, 0x0C39);
        addRange(LCat, 0x0C41, 0x0C44);
        addRange(LCat, 0x0C60, 0x0C61);
        addRange(LCat, 0x0C66, 0x0C6F);
        addRange(LCat, 0x0C82, 0x0C83);
        addRange(LCat, 0x0C85, 0x0C8C);
        addRange(LCat, 0x0C8E, 0x0C90);
        addRange(LCat, 0x0C92, 0x0CA8);
        addRange(LCat, 0x0CAA, 0x0CB3);
        addRange(LCat, 0x0CB5, 0x0CB9);
        LCat.add("\u0CBE");
        addRange(LCat, 0x0CC0, 0x0CC4);
        addRange(LCat, 0x0CC7, 0x0CC8);
        addRange(LCat, 0x0CCA, 0x0CCB);
        addRange(LCat, 0x0CD5, 0x0CD6);
        LCat.add("\u0CDE");
        addRange(LCat, 0x0CE0, 0x0CE1);
        addRange(LCat, 0x0CE6, 0x0CEF);
        addRange(LCat, 0x0D02, 0x0D03);
        addRange(LCat, 0x0D05, 0x0D0C);
        addRange(LCat, 0x0D0E, 0x0D10);
        addRange(LCat, 0x0D12, 0x0D28);
        addRange(LCat, 0x0D2A, 0x0D39);
        addRange(LCat, 0x0D3E, 0x0D40);
        addRange(LCat, 0x0D46, 0x0D48);
        addRange(LCat, 0x0D4A, 0x0D4C);
        LCat.add("\u0D57");
        addRange(LCat, 0x0D60, 0x0D61);
        addRange(LCat, 0x0D66, 0x0D6F);
        addRange(LCat, 0x0D82, 0x0D83);
        addRange(LCat, 0x0D85, 0x0D96);
        addRange(LCat, 0x0D9A, 0x0DB1);
        addRange(LCat, 0x0DB3, 0x0DBB);
        LCat.add("\u0DBD");
        addRange(LCat, 0x0DC0, 0x0DC6);
        addRange(LCat, 0x0DCF, 0x0DD1);
        addRange(LCat, 0x0DD8, 0x0DDF);
        addRange(LCat, 0x0DF2, 0x0DF4);
        addRange(LCat, 0x0E01, 0x0E30);
        addRange(LCat, 0x0E32, 0x0E33);
        addRange(LCat, 0x0E40, 0x0E46);
        addRange(LCat, 0x0E4F, 0x0E5B);
        addRange(LCat, 0x0E81, 0x0E82);
        LCat.add("\u0E84");
        addRange(LCat, 0x0E87, 0x0E88);
        LCat.add("\u0E8A");
        LCat.add("\u0E8D");
        addRange(LCat, 0x0E94, 0x0E97);
        addRange(LCat, 0x0E99, 0x0E9F);
        addRange(LCat, 0x0EA1, 0x0EA3);
        LCat.add("\u0EA5");
        LCat.add("\u0EA7");
        addRange(LCat, 0x0EAA, 0x0EAB);
        addRange(LCat, 0x0EAD, 0x0EB0);
        addRange(LCat, 0x0EB2, 0x0EB3);
        LCat.add("\u0EBD");
        addRange(LCat, 0x0EC0, 0x0EC4);
        LCat.add("\u0EC6");
        addRange(LCat, 0x0ED0, 0x0ED9);
        addRange(LCat, 0x0EDC, 0x0EDD);
        addRange(LCat, 0x0F00, 0x0F17);
        addRange(LCat, 0x0F1A, 0x0F34);
        LCat.add("\u0F36");
        LCat.add("\u0F38");
        addRange(LCat, 0x0F3E, 0x0F47);
        addRange(LCat, 0x0F49, 0x0F6A);
        LCat.add("\u0F7F");
        LCat.add("\u0F85");
        addRange(LCat, 0x0F88, 0x0F8B);
        addRange(LCat, 0x0FBE, 0x0FC5);
        addRange(LCat, 0x0FC7, 0x0FCC);
        LCat.add("\u0FCF");
        addRange(LCat, 0x1000, 0x1021);
        addRange(LCat, 0x1023, 0x1027);
        addRange(LCat, 0x1029, 0x102A);
        LCat.add("\u102C");
        LCat.add("\u1031");
        LCat.add("\u1038");
        addRange(LCat, 0x1040, 0x1057);
        addRange(LCat, 0x10A0, 0x10C5);
        addRange(LCat, 0x10D0, 0x10F8);
        LCat.add("\u10FB");
        addRange(LCat, 0x1100, 0x1159);
        addRange(LCat, 0x115F, 0x11A2);
        addRange(LCat, 0x11A8, 0x11F9);
        addRange(LCat, 0x1200, 0x1206);
        addRange(LCat, 0x1208, 0x1246);
        LCat.add("\u1248");
        addRange(LCat, 0x124A, 0x124D);
        addRange(LCat, 0x1250, 0x1256);
        LCat.add("\u1258");
        addRange(LCat, 0x125A, 0x125D);
        addRange(LCat, 0x1260, 0x1286);
        LCat.add("\u1288");
        addRange(LCat, 0x128A, 0x128D);
        addRange(LCat, 0x1290, 0x12AE);
        LCat.add("\u12B0");
        addRange(LCat, 0x12B2, 0x12B5);
        addRange(LCat, 0x12B8, 0x12BE);
        LCat.add("\u12C0");
        addRange(LCat, 0x12C2, 0x12C5);
        addRange(LCat, 0x12C8, 0x12CE);
        addRange(LCat, 0x12D0, 0x12D6);
        addRange(LCat, 0x12D8, 0x12EE);
        addRange(LCat, 0x12F0, 0x130E);
        LCat.add("\u1310");
        addRange(LCat, 0x1312, 0x1315);
        addRange(LCat, 0x1318, 0x131E);
        addRange(LCat, 0x1320, 0x1346);
        addRange(LCat, 0x1348, 0x135A);
        addRange(LCat, 0x1361, 0x137C);
        addRange(LCat, 0x13A0, 0x13F4);
        addRange(LCat, 0x1401, 0x1676);
        addRange(LCat, 0x1681, 0x169A);
        addRange(LCat, 0x16A0, 0x16F0);
        addRange(LCat, 0x1700, 0x170C);
        addRange(LCat, 0x170E, 0x1711);
        addRange(LCat, 0x1720, 0x1731);
        addRange(LCat, 0x1735, 0x1736);
        addRange(LCat, 0x1740, 0x1751);
        addRange(LCat, 0x1760, 0x176C);
        addRange(LCat, 0x176E, 0x1770);
        addRange(LCat, 0x1780, 0x17B6);
        addRange(LCat, 0x17BE, 0x17C5);
        addRange(LCat, 0x17C7, 0x17C8);
        addRange(LCat, 0x17D4, 0x17DA);
        LCat.add("\u17DC");
        addRange(LCat, 0x17E0, 0x17E9);
        addRange(LCat, 0x1810, 0x1819);
        addRange(LCat, 0x1820, 0x1877);
        addRange(LCat, 0x1880, 0x18A8);
        addRange(LCat, 0x1E00, 0x1E9B);
        addRange(LCat, 0x1EA0, 0x1EF9);
        addRange(LCat, 0x1F00, 0x1F15);
        addRange(LCat, 0x1F18, 0x1F1D);
        addRange(LCat, 0x1F20, 0x1F45);
        addRange(LCat, 0x1F48, 0x1F4D);
        addRange(LCat, 0x1F50, 0x1F57);
        LCat.add("\u1F59");
        LCat.add("\u1F5B");
        LCat.add("\u1F5D");
        addRange(LCat, 0x1F5F, 0x1F7D);
        addRange(LCat, 0x1F80, 0x1FB4);
        addRange(LCat, 0x1FB6, 0x1FBC);
        LCat.add("\u1FBE");
        addRange(LCat, 0x1FC2, 0x1FC4);
        addRange(LCat, 0x1FC6, 0x1FCC);
        addRange(LCat, 0x1FD0, 0x1FD3);
        addRange(LCat, 0x1FD6, 0x1FDB);
        addRange(LCat, 0x1FE0, 0x1FEC);
        addRange(LCat, 0x1FF2, 0x1FF4);
        addRange(LCat, 0x1FF6, 0x1FFC);
        LCat.add("\u200E");
        LCat.add("\u2071");
        LCat.add("\u207F");
        LCat.add("\u2102");
        LCat.add("\u2107");
        addRange(LCat, 0x210A, 0x2113);
        LCat.add("\u2115");
        addRange(LCat, 0x2119, 0x211D);
        LCat.add("\u2124");
        LCat.add("\u2126");
        LCat.add("\u2128");
        addRange(LCat, 0x212A, 0x212D);
        addRange(LCat, 0x212F, 0x2131);
        addRange(LCat, 0x2133, 0x2139);
        addRange(LCat, 0x213D, 0x213F);
        addRange(LCat, 0x2145, 0x2149);
        addRange(LCat, 0x2160, 0x2183);
        addRange(LCat, 0x2336, 0x237A);
        LCat.add("\u2395");
        addRange(LCat, 0x249C, 0x24E9);
        addRange(LCat, 0x3005, 0x3007);
        addRange(LCat, 0x3021, 0x3029);
        addRange(LCat, 0x3031, 0x3035);
        addRange(LCat, 0x3038, 0x303C);
        addRange(LCat, 0x3041, 0x3096);
        addRange(LCat, 0x309D, 0x309F);
        addRange(LCat, 0x30A1, 0x30FA);
        addRange(LCat, 0x30FC, 0x30FF);
        addRange(LCat, 0x3105, 0x312C);
        addRange(LCat, 0x3131, 0x318E);
        addRange(LCat, 0x3190, 0x31B7);
        addRange(LCat, 0x31F0, 0x321C);
        addRange(LCat, 0x3220, 0x3243);
        addRange(LCat, 0x3260, 0x327B);
        addRange(LCat, 0x327F, 0x32B0);
        addRange(LCat, 0x32C0, 0x32CB);
        addRange(LCat, 0x32D0, 0x32FE);
        addRange(LCat, 0x3300, 0x3376);
        addRange(LCat, 0x337B, 0x33DD);
        addRange(LCat, 0x33E0, 0x33FE);
        addRange(LCat, 0x3400, 0x4DB5);
        addRange(LCat, 0x4E00, 0x9FA5);
        addRange(LCat, 0xA000, 0xA48C);
        addRange(LCat, 0xAC00, 0xD7A3);
        addRange(LCat, 0xD800, 0xFA2D);
        addRange(LCat, 0xFA30, 0xFA6A);
        addRange(LCat, 0xFB00, 0xFB06);
        addRange(LCat, 0xFB13, 0xFB17);
        addRange(LCat, 0xFF21, 0xFF3A);
        addRange(LCat, 0xFF41, 0xFF5A);
        addRange(LCat, 0xFF66, 0xFFBE);
        addRange(LCat, 0xFFC2, 0xFFC7);
        addRange(LCat, 0xFFCA, 0xFFCF);
        addRange(LCat, 0xFFD2, 0xFFD7);
        addRange(LCat, 0xFFDA, 0xFFDC);
        addRange(LCat, 0x10300, 0x1031E);
        addRange(LCat, 0x10320, 0x10323);
        addRange(LCat, 0x10330, 0x1034A);
        addRange(LCat, 0x10400, 0x10425);
        addRange(LCat, 0x10428, 0x1044D);
        addRange(LCat, 0x1D000, 0x1D0F5);
        addRange(LCat, 0x1D100, 0x1D126);
        addRange(LCat, 0x1D12A, 0x1D166);
        addRange(LCat, 0x1D16A, 0x1D172);
        addRange(LCat, 0x1D183, 0x1D184);
        addRange(LCat, 0x1D18C, 0x1D1A9);
        addRange(LCat, 0x1D1AE, 0x1D1DD);
        addRange(LCat, 0x1D400, 0x1D454);
        addRange(LCat, 0x1D456, 0x1D49C);
        addRange(LCat, 0x1D49E, 0x1D49F);
        LCat.add(codePointToString(0x1D4A2));
        addRange(LCat, 0x1D4A5, 0x1D4A6);
        addRange(LCat, 0x1D4A9, 0x1D4AC);
        addRange(LCat, 0x1D4AE, 0x1D4B9);
        LCat.add(codePointToString(0x1D4BB));
        addRange(LCat, 0x1D4BD, 0x1D4C0);
        addRange(LCat, 0x1D4C2, 0x1D4C3);
        addRange(LCat, 0x1D4C5, 0x1D505);
        addRange(LCat, 0x1D507, 0x1D50A);
        addRange(LCat, 0x1D50D, 0x1D514);
        addRange(LCat, 0x1D516, 0x1D51C);
        addRange(LCat, 0x1D51E, 0x1D539);
        addRange(LCat, 0x1D53B, 0x1D53E);
        addRange(LCat, 0x1D540, 0x1D544);
        LCat.add(codePointToString(0x1D546));
        addRange(LCat, 0x1D54A, 0x1D550);
        addRange(LCat, 0x1D552, 0x1D6A3);
        addRange(LCat, 0x1D6A8, 0x1D7C9);
        addRange(LCat, 0x20000, 0x2A6D6);
        addRange(LCat, 0x2F800, 0x2FA1D);
        addRange(LCat, 0xF0000, 0xFFFFD);
        addRange(LCat, 0x100000, 0x10FFFD);
        return Collections.unmodifiableSet(LCat);
    }

    private static final void addRange(Set<String> set, int start, int stop) {
        for (int i = start; i <= stop; i++) {
            String s = new String(Character.toChars(i));
            set.add(s);
        }
    }

    private static final String codePointToString(int codePoint) {
        return new String(Character.toChars(codePoint));
    }

}
