/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker.unsafequeues;

import io.moquette.broker.unsafequeues.PagedFilesAllocator;
import io.moquette.broker.unsafequeues.Queue;
import io.moquette.broker.unsafequeues.QueueException;
import io.moquette.broker.unsafequeues.Segment;
import io.moquette.broker.unsafequeues.SegmentAllocator;
import io.moquette.broker.unsafequeues.SegmentPointer;
import io.moquette.broker.unsafequeues.VirtualPointer;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueuePool {
    private static final Logger LOG = LoggerFactory.getLogger(QueuePool.class);
    static final boolean queueDebug = Boolean.parseBoolean(System.getProperty("moquette.queue.debug", "false"));
    private final SegmentAllocationCallback callback;
    private final SegmentAllocator allocator;
    private final Path dataPath;
    private final int segmentSize;
    private final ConcurrentMap<QueueName, LinkedList<SegmentRef>> queueSegments = new ConcurrentHashMap<QueueName, LinkedList<SegmentRef>>();
    private final ConcurrentMap<QueueName, Queue> queues = new ConcurrentHashMap<QueueName, Queue>();
    private final ConcurrentSkipListSet<SegmentRef> recycledSegments = new ConcurrentSkipListSet();
    private final ReentrantLock segmentsAllocationLock = new ReentrantLock();

    private QueuePool(SegmentAllocator allocator, Path dataPath, int segmentSize) {
        this.allocator = allocator;
        this.dataPath = dataPath;
        this.segmentSize = segmentSize;
        this.callback = new SegmentAllocationCallback(this);
    }

    private void segmentedCreated(String name, Segment segment) {
        LOG.debug("Registering new segment {} for queue {}", (Object)segment, (Object)name);
        QueueName queueName = new QueueName(name);
        List segmentRefs = this.queueSegments.computeIfAbsent(queueName, k -> new LinkedList());
        segmentRefs.add(0, new SegmentRef(segment));
        LOG.debug("queueSegments for queue {} after insertion {}", (Object)queueName, (Object)segmentRefs);
    }

    public static QueuePool loadQueues(Path dataPath, int pageSize, int segmentSize) throws QueueException {
        Properties checkpointProps = QueuePool.createOrLoadCheckpointFile(dataPath);
        int lastPage = Integer.parseInt(checkpointProps.getProperty("segments.last_page", "0"));
        int lastSegment = Integer.parseInt(checkpointProps.getProperty("segments.last_segment", "0"));
        PagedFilesAllocator allocator = new PagedFilesAllocator(dataPath, pageSize, segmentSize, lastPage, lastSegment);
        QueuePool queuePool = new QueuePool(allocator, dataPath, segmentSize);
        queuePool.loadQueueDefinitions(checkpointProps);
        LOG.debug("Loaded queues definitions: {}", queuePool.queueSegments);
        queuePool.loadRecycledSegments(checkpointProps);
        LOG.debug("Recyclable segments are: {}", queuePool.recycledSegments);
        return queuePool;
    }

    public Set<String> queueNames() {
        return this.queues.keySet().stream().map(qn -> qn.name).collect(Collectors.toSet());
    }

    private static Properties createOrLoadCheckpointFile(Path dataPath) throws QueueException {
        FileReader fileReader;
        Path checkpointPath = dataPath.resolve("checkpoint.properties");
        if (!Files.exists(checkpointPath, new LinkOption[0])) {
            boolean notExisted;
            LOG.info("Can't find any file named 'checkpoint.properties' in path: {}, creating new one", (Object)dataPath);
            try {
                notExisted = checkpointPath.toFile().createNewFile();
            }
            catch (IOException e) {
                LOG.error("IO Error creating the file {}", (Object)checkpointPath, (Object)e);
                throw new QueueException("Reached an IO error during the bootstrapping of empty 'checkpoint.properties'", e);
            }
            if (!notExisted) {
                LOG.warn("Found a checkpoint file while bootstrapping {}", (Object)checkpointPath);
            }
        }
        try {
            fileReader = new FileReader(checkpointPath.toFile());
        }
        catch (FileNotFoundException e) {
            throw new QueueException("Can't find any file named 'checkpoint.properties' in path: " + dataPath, e);
        }
        Properties checkpointProps = new Properties();
        try {
            checkpointProps.load(fileReader);
        }
        catch (IOException e) {
            throw new QueueException("if an error occurred when reading from: " + checkpointPath, e);
        }
        return checkpointProps;
    }

    private void loadQueueDefinitions(Properties checkpointProps) throws QueueException {
        boolean noMoreQueues = false;
        int queueId = 0;
        while (!noMoreQueues) {
            String queueKey = String.format("queues.%d.name", queueId);
            if (!checkpointProps.containsKey(queueKey)) {
                noMoreQueues = true;
                continue;
            }
            QueueName queueName = new QueueName(checkpointProps.getProperty(queueKey));
            LinkedList<SegmentRef> segmentRefs = this.decodeSegments(checkpointProps.getProperty(String.format("queues.%d.segments", queueId)));
            int numSegments = segmentRefs.size();
            this.queueSegments.put(queueName, segmentRefs);
            long headOffset = Long.parseLong(checkpointProps.getProperty(String.format("queues.%d.head_offset", queueId)));
            SegmentRef headSegmentRef = segmentRefs.get(0);
            SegmentPointer currentHead = new SegmentPointer(headSegmentRef.pageId, headOffset);
            Segment headSegment = this.allocator.reopenSegment(headSegmentRef.pageId, headSegmentRef.offset);
            long tailOffset = Long.parseLong(checkpointProps.getProperty(String.format("queues.%d.tail_offset", queueId)));
            SegmentRef tailSegmentRef = segmentRefs.getLast();
            SegmentPointer currentTail = new SegmentPointer(tailSegmentRef.pageId, tailOffset);
            Segment tailSegment = this.allocator.reopenSegment(tailSegmentRef.pageId, tailSegmentRef.offset);
            VirtualPointer logicalTail = new VirtualPointer(currentTail.offset());
            VirtualPointer logicalHead = new VirtualPointer((long)(numSegments - 1) * (long)this.segmentSize + (long)currentHead.offset());
            Queue queue = new Queue(queueName.name, headSegment, logicalHead, tailSegment, logicalTail, this.allocator, this.callback, this);
            this.queues.put(queueName, queue);
            ++queueId;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadRecycledSegments(Properties checkpointProps) throws QueueException {
        TreeSet<SegmentRef> usedSegments = new TreeSet<SegmentRef>();
        boolean noMoreQueues = false;
        int queueId = 0;
        while (!noMoreQueues) {
            String queueKey = String.format("queues.%d.name", queueId);
            if (!checkpointProps.containsKey(queueKey)) {
                noMoreQueues = true;
                continue;
            }
            LinkedList<SegmentRef> segmentRefs = this.decodeSegments(checkpointProps.getProperty(String.format("queues.%d.segments", queueId)));
            usedSegments.addAll(segmentRefs);
            ++queueId;
        }
        if (usedSegments.isEmpty()) {
            return;
        }
        List<SegmentRef> recreatedSegments = this.recreateSegmentHoles(usedSegments);
        this.segmentsAllocationLock.lock();
        try {
            this.recycledSegments.addAll(recreatedSegments);
        }
        finally {
            this.segmentsAllocationLock.unlock();
        }
    }

    List<SegmentRef> recreateSegmentHoles(TreeSet<SegmentRef> usedSegments) throws QueueException {
        if (usedSegments.isEmpty()) {
            throw new QueueException("Status error, expected to find at least one segment");
        }
        SegmentRef prev = null;
        LinkedList<SegmentRef> recreatedSegments = new LinkedList<SegmentRef>();
        for (SegmentRef current : usedSegments) {
            if (prev == null) {
                recreatedSegments.addAll(this.recreateRecycledSegmentsBetween(current));
                prev = current;
                continue;
            }
            if (this.isAdjacent(prev, current)) {
                prev = current;
                continue;
            }
            if (prev.pageId == current.pageId) {
                recreatedSegments.addAll(this.recreateRecycledSegments(prev.offset + this.segmentSize, current.offset, prev.pageId));
                continue;
            }
            recreatedSegments.addAll(this.recreateRecycledSegmentsBetween(prev, current));
        }
        return recreatedSegments;
    }

    private boolean isAdjacent(SegmentRef prev, SegmentRef segment) {
        return prev.pageId == segment.pageId ? prev.offset + this.segmentSize == segment.offset : prev.pageId + 1 == segment.pageId && prev.offset == this.allocator.getPageSize() - this.segmentSize && segment.offset == 0;
    }

    private List<SegmentRef> recreateRecycledSegmentsBetween(SegmentRef toSegment) {
        return this.recreateRecycledSegmentsBetween(null, toSegment);
    }

    private List<SegmentRef> recreateRecycledSegmentsBetween(SegmentRef fromSegment, SegmentRef toSegment) {
        LinkedList<SegmentRef> recreatedSegments = new LinkedList<SegmentRef>();
        int prevPageId = 0;
        if (fromSegment != null) {
            prevPageId = fromSegment.pageId;
            recreatedSegments.addAll(this.recreateRecycledSegments(fromSegment.offset + this.segmentSize, this.allocator.getPageSize(), fromSegment.pageId));
            ++prevPageId;
        }
        while (prevPageId < toSegment.pageId) {
            recreatedSegments.addAll(this.recreateRecycledSegments(0, this.allocator.getPageSize(), prevPageId));
            ++prevPageId;
        }
        recreatedSegments.addAll(this.recreateRecycledSegments(0, toSegment.offset, toSegment.pageId));
        return recreatedSegments;
    }

    private List<SegmentRef> recreateRecycledSegments(int fromOffset, int toOffset, int pageId) {
        LinkedList<SegmentRef> recreatedSegments = new LinkedList<SegmentRef>();
        while (fromOffset != toOffset) {
            recreatedSegments.add(new SegmentRef(pageId, fromOffset));
            fromOffset += this.segmentSize;
        }
        return recreatedSegments;
    }

    private LinkedList<SegmentRef> decodeSegments(String s) {
        String[] segments = s.substring(s.indexOf("(") + 1, s.lastIndexOf(")")).split("\\), \\(");
        LinkedList<SegmentRef> acc = new LinkedList<SegmentRef>();
        for (String segment : segments) {
            String[] split = segment.split(",");
            int idPage = Integer.parseInt(split[0].trim());
            int offset = Integer.parseInt(split[1].trim());
            acc.offer(new SegmentRef(idPage, offset));
        }
        return acc;
    }

    public Queue getOrCreate(String queueName) throws QueueException {
        QueueName queueN = new QueueName(queueName);
        if (this.queues.containsKey(queueN)) {
            return (Queue)this.queues.get(queueN);
        }
        Segment segment = this.nextFreeSegment();
        this.segmentedCreated(queueName, segment);
        Queue queue = new Queue(queueName, segment, VirtualPointer.buildUntouched(), segment, VirtualPointer.buildUntouched(), this.allocator, this.callback, this);
        this.queues.put(queueN, queue);
        return queue;
    }

    public void close() throws QueueException {
        FileWriter fileWriter;
        this.allocator.close();
        Properties checkpoint = new Properties();
        this.allocator.dumpState(checkpoint);
        int queueCounter = 0;
        for (Map.Entry entry : this.queueSegments.entrySet()) {
            QueueName queueName = (QueueName)entry.getKey();
            checkpoint.setProperty("queues." + queueCounter + ".name", queueName.name);
            LinkedList segmentRefs = (LinkedList)entry.getValue();
            String segmentsDef = segmentRefs.stream().map(SegmentRef::toString).collect(Collectors.joining(", "));
            checkpoint.setProperty("queues." + queueCounter + ".segments", segmentsDef);
            Queue queue = (Queue)this.queues.get(queueName);
            checkpoint.setProperty("queues." + queueCounter + ".head_offset", String.valueOf(queue.currentHead().segmentOffset(this.segmentSize)));
            checkpoint.setProperty("queues." + queueCounter + ".tail_offset", String.valueOf(queue.currentTail().segmentOffset(this.segmentSize)));
        }
        File propertiesFile = this.dataPath.resolve("checkpoint.properties").toFile();
        try {
            fileWriter = new FileWriter(propertiesFile);
        }
        catch (IOException ex) {
            throw new QueueException("Problem opening checkpoint.properties file", ex);
        }
        try {
            checkpoint.store(fileWriter, "DON'T EDIT, AUTOGENERATED");
        }
        catch (IOException ex) {
            throw new QueueException("Problem writing checkpoint.properties file", ex);
        }
    }

    Optional<Segment> openNextTailSegment(String name) throws QueueException {
        MappedByteBuffer tailPage;
        QueueName queueName = new QueueName(name);
        LinkedList segmentRefs = (LinkedList)this.queueSegments.get(queueName);
        SegmentRef pollSegment = (SegmentRef)segmentRefs.peekLast();
        if (pollSegment == null) {
            return Optional.empty();
        }
        Path pageFile = this.dataPath.resolve(String.format("%d.page", pollSegment.pageId));
        if (!Files.exists(pageFile, new LinkOption[0])) {
            throw new QueueException("Can't find file for page file" + pageFile);
        }
        try (FileChannel fileChannel = FileChannel.open(pageFile, StandardOpenOption.READ, StandardOpenOption.WRITE);){
            tailPage = fileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, this.allocator.getPageSize());
        }
        catch (IOException ex) {
            throw new QueueException("Can't open page file " + pageFile, ex);
        }
        SegmentPointer begin = new SegmentPointer(pollSegment.pageId, (long)pollSegment.offset);
        SegmentPointer end = new SegmentPointer(pollSegment.pageId, (long)(pollSegment.offset + this.segmentSize - 1));
        return Optional.of(new Segment(tailPage, begin, end));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void consumedTailSegment(String name) {
        QueueName queueName = new QueueName(name);
        LinkedList segmentRefs = (LinkedList)this.queueSegments.get(queueName);
        SegmentRef segmentRef = (SegmentRef)segmentRefs.pollLast();
        LOG.debug("Consumed tail segment {} from queue {}", (Object)segmentRef, (Object)queueName);
        this.segmentsAllocationLock.lock();
        try {
            this.recycledSegments.add(segmentRef);
        }
        finally {
            this.segmentsAllocationLock.unlock();
        }
    }

    Segment nextFreeSegment() throws QueueException {
        this.segmentsAllocationLock.lock();
        try {
            if (this.recycledSegments.isEmpty()) {
                LOG.debug("no recycled segments available, request the creation of new one");
                Segment segment = this.allocator.nextFreeSegment();
                return segment;
            }
            SegmentRef recycledSegment = this.recycledSegments.pollFirst();
            if (recycledSegment == null) {
                throw new QueueException("Invalid state, expected available recycled segment");
            }
            LOG.debug("Reusing recycled segment from page: {} at page offset: {}", (Object)recycledSegment.pageId, (Object)recycledSegment.offset);
            Segment segment = this.allocator.reopenSegment(recycledSegment.pageId, recycledSegment.offset);
            return segment;
        }
        finally {
            this.segmentsAllocationLock.unlock();
        }
    }

    private static class SegmentAllocationCallback
    implements PagedFilesAllocator.AllocationListener {
        private final QueuePool queuePool;

        private SegmentAllocationCallback(QueuePool queuePool) {
            this.queuePool = queuePool;
        }

        @Override
        public void segmentedCreated(String name, Segment segment) {
            this.queuePool.segmentedCreated(name, segment);
        }
    }

    private static class QueueName {
        final String name;

        private QueueName(String name) {
            this.name = name;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            QueueName queueName = (QueueName)o;
            return Objects.equals(this.name, queueName.name);
        }

        public int hashCode() {
            return Objects.hash(this.name);
        }

        public String toString() {
            return "QueueName{name='" + this.name + '\'' + '}';
        }
    }

    static class SegmentRef
    implements Comparable<SegmentRef> {
        final int pageId;
        final int offset;

        SegmentRef(int pageId, int offset) {
            this.pageId = pageId;
            this.offset = offset;
        }

        public SegmentRef(Segment segment) {
            this.pageId = segment.begin.pageId();
            this.offset = segment.begin.offset();
        }

        public String toString() {
            return String.format("(%d, %d)", this.pageId, this.offset);
        }

        @Override
        public int compareTo(SegmentRef o) {
            int pageCompare = Integer.compare(this.pageId, o.pageId);
            if (pageCompare != 0) {
                return pageCompare;
            }
            return Integer.compare(this.offset, o.offset);
        }
    }
}

